
// quote.cpp
// Copyright 2015 Matthew Rickard

// This file is part of dep
// dep is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <https://www.gnu.org/licenses/>.

#include "precomp.h"
#include "quote.h"
#include "logMsg.h"
#include <string.h>

static char nibble(unsigned char x) {
  if (x < 10)
    return '0' + x;
  else
    return 'a' + x - 10;
}

std::string hexPair(unsigned char x) {
  char a[3];
  a[0] = nibble(x >> 4);
  a[1] = nibble(x & 15);
  a[2] = 0;
  return std::string(a);
}

std::string quote(const char *s, int len) {
  std::string result;
  for (int i = 0; i < len; i++) {
    if (s[i] >= 32 && s[i] <= 126)
      result.append(1, s[i]);
    else if (s[i] == '\a') result += ("\\a");
    else if (s[i] == '\b') result += ("\\b");
    else if (s[i] == '\f') result += ("\\f");
    else if (s[i] == '\n') result += ("\\n");
    else if (s[i] == '\r') result += ("\\r");
    else if (s[i] == '\t') result += ("\\t");
    else if (s[i] == '\v') result += ("\\v");
    else
      result.append("\\0x").append(hexPair(s[i]));
  }
  return result;
}

std::string partlyQuote(const char *s, int len) {
  std::string result;
  for (int i = 0; i < len; i++) {
    if (s[i] >= 32 && s[i] <= 126)
      result.append(1, s[i]);
    else if (s[i] == '\a') result += ("\\a");
    else if (s[i] == '\b') result += ("\\b");
    else if (s[i] == '\f') result += ("\\f");
    else if (s[i] == '\n') result += ("\n");
    else if (s[i] == '\r') result += ("\\r");
    else if (s[i] == '\t') result += ("\\t");
    else if (s[i] == '\v') result += ("\\v");
    else
      result.append("\\0x").append(hexPair(s[i]));
  }
  return result;
}

std::string quote(char s) {
  std::string s2;
  s2.push_back(s);
  return quote(s2);
}

std::string partlyQuote(const std::string &s) {
  return partlyQuote(s.c_str(), s.size());
}

std::string quote(const std::string &s) {
  return quote(s.c_str(), s.size());
}

std::string hexquote(const std::string &s) {
  return hexquote(s.c_str(), s.size());
}

std::string quote(const char *s) {
  return quote(s, strlen(s));
}

std::string hexquote(const char *s, int len) {
  std::string result;
  for (int i = 0; i < len; i++)
    result.append("\\0x").append(hexPair(s[i]));
  //logMsg << logValue(result) << '.' << std::endl;
  return result;
}

std::string hexquote(const wchar_t *s, int len) {
  return hexquote((const char *)s, 2 * len);
}

std::string hexquote(const char *s) {
  return hexquote(s, strlen(s));
}

std::string hexquote(const wchar_t *s) {
  return hexquote((const char *)s, 2 * wcslen(s));
}

std::string hexquote(const std::wstring &s) {
  return hexquote((const char *)s.c_str(), 2 * s.size());
}

std::string strings(const std::string &s) {
  return strings(s.c_str(), s.size());
}

std::string strings(const char *s, int len) {
  std::string result, tmp;
  const int stringLength = 7;
  for (int i = 0; i < len; i++) {
    if (s[i] >= 32 && s[i] <= 126)
      tmp.append(1, s[i]);
    else {
      if (tmp.size() >= stringLength) {
        result.append(tmp);
        result.append("\n");
      }
      tmp.resize(0);
    }
  }
  if (tmp.size() >= stringLength) {
    result.append(tmp);
    result.append("\n");
  }
  return result;
}

std::string quoteAsNec(String s) {
  if (!*s.s)
    return "\"\"";
  if (s.s[0] == '\'' || s.s[0] == '\"')
    return s.s;
  if (strchr(s.s, ' '))
    return std::string("\"") + s.s + "\"";
  return s.s;
}

