
// interproc.h
// Copyright 2024 Matthew Rickard
// This file is part of dep

// dep is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <https://www.gnu.org/licenses/>.

#ifndef INTERPROC_H
#define INTERPROC_H

#if defined _WIN2 || defined __CYGWIN__ || defined __MINGW32__
#include <windows.h>
#undef min
#undef max
#endif
#include <pthread.h>
#include <atomic>

/* Which mutex? There are five possibilities:
   C11 mtx_t              Apparently no-one uses this
   POSIX pthread_mutex_t  Works on linux. Documentation is lousy.
                          PTHREAD_PROCESS_SHARED is unsupported on Cygwin
   C++11 std::mutex       Wrapper around pthread_mutex_t. Probably based on boost::mutex.
                          Supplies native_handle
   boost::mutex           Based on pthread_mutex_t. Probably a wrapper. Supplies native_handle
                          A rule of thumb: if it is in STD C++, don't use Boost
   Win32                  Probably works on windows

*/

// Powers of two
#define LOCK_GUI 1
#define LOCK_DB  2

#define MAX_LOCKS 2 // the number of them

struct Mutex {
#if defined _WIN2 || defined __CYGWIN__ || defined __MINGW32__
  HANDLE mutex;
#else
  pthread_mutex_t mutex;
#endif

  void initialize();
  int lock(int timeoutMs, int *mutexFlags = 0);
  void unlock();
};

#define INTERPROC_OK         11
#define INTERPROC_TIMEOUT    12
#define INTERPROC_FAILED     13

#define MUTEXFLAGS_ABANDONED 16

struct Event {
#if defined _WIN2 || defined __CYGWIN__ || defined __MINGW32__
  HANDLE event;
#else
  int event;
#endif

  void initialize();
  int setEvent();
  int wait(int timeoutMs);
};

struct CountingMutex {
  Mutex *mutex;
  int counter;

  void operator = (Mutex &);
  int lock(int timeoutMs, int *mutexFlags = 0);
  void unlock();
};

struct Lock {
  Mutex *mutex;
  int locked;

  Lock(Mutex &mutex, int timeoutMs);
  void unlock();
  ~Lock();
};

struct GlobalShare {
  short stopnow;
  std::atomic<short> workersActive;
  Mutex statsLock;
  double weightedRequestSum;
  double weightSum;
  Mutex mutex[MAX_LOCKS];

  void initialize();
};

struct InterProc {
  std::atomic<int> passes, fails, tests;
  std::atomic<long long> heap, stack;
  Event workerStartedEvent;
  Event resumeWorkerEvent;
  std::atomic<short> pausedWorkers;
  std::atomic<short> pauseRequests;

  void initialize();
};

void makeTimeoutMillis(timespec *ts, int millis);

extern GlobalShare *globalShare;
extern InterProc *interproc;

#endif

