
// interproc.cpp
// Copyright 2024 Matthew Rickard
// This file is part of dep

// dep is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <https://www.gnu.org/licenses/>.

#include "precomp.h"
#include "interproc.h"
#include "logMsg.h"
#include "MacroNames.h"
#include "argv.h"
#if !defined _WIN2 && !defined __CYGWIN__ && !defined __MINGW32__
#include <sys/eventfd.h>
#include <unistd.h>
#endif

void InterProc::initialize() {
  passes = 0;
  fails = 0;
  tests = 0;
  heap = 0;
  stack = 0;
  pausedWorkers = 0;
  pauseRequests = 0;
//#if defined _WIN2 || defined __CYGWIN__ || defined __MINGW32__
  workerStartedEvent.initialize();
  resumeWorkerEvent.initialize();
//#else
  //lineabort();
//#endif
}

void GlobalShare::initialize() {
  //logMsg << "Initializing GlobalShare" << std::endl;
  stopnow = 0;
  workersActive = 0;
  statsLock.initialize();
  weightedRequestSum = 0;
  weightSum = 0;
  for (int i = 0; i < MAX_LOCKS; i++) {
    mutex[i].initialize();
  }
}

#if defined _WIN2 || defined __CYGWIN__ || defined __MINGW32__
static SECURITY_ATTRIBUTES sa = { sizeof sa, 0, 1 };
#endif

void Mutex::initialize() {
#if defined _WIN2 || defined __CYGWIN__ || defined __MINGW32__
  mutex = CreateMutexEx(&sa, 0, 0, DELETE | SYNCHRONIZE);
#else
  pthread_mutexattr_t attr;
  int result = pthread_mutexattr_init(&attr);
  if (result)
    logMsg << logValue(result) << std::endl;
  result = pthread_mutexattr_setpshared(&attr, PTHREAD_PROCESS_SHARED);
  if (result)
    logMsg << logValue(result) << ' ' << ErrnoToMacroName(result) << std::endl;
  result = pthread_mutexattr_setrobust(&attr, PTHREAD_MUTEX_ROBUST);
  if (result)
    logMsg << logValue(result) << ' ' << ErrnoToMacroName(result) << std::endl;
  result = pthread_mutex_init(&mutex, &attr);
  if (result)
      logMsg << logValue(result) << std::endl;
#endif
}

// On WinAPI, INFINITE is 0xffffffff which in an int (4 bytes) is -1

void makeTimeoutMillis(timespec *ts, int millis) {
  clock_gettime(CLOCK_REALTIME, ts);
  //logMsg << logValue(ts.tv_sec) << ' ' << logValue(ts.tv_nsec) << std::endl;
  if (millis > 999) {
    ts->tv_sec += millis / 1000;
    ts->tv_nsec += (millis % 1000) * 1000000;
  }
  else
    ts->tv_nsec += millis * 1000000;
  if (ts->tv_nsec >= 1000000000) {
    //logMsg << logValue(ts.tv_sec) << ' ' << logValue(ts.tv_nsec) << std::endl;
    ts->tv_nsec -= 1000000000;
    ts->tv_sec += 1;
  }
}

int Mutex::lock(int timeoutMs, int *mutexFlags /* = 0 */) {
  int result;
  //logMsg << olib::argv[0] << " locking mutex" << std::endl;
#if defined _WIN2 || defined __CYGWIN__ || defined __MINGW32__
  unsigned waitresult = WaitForSingleObject(mutex, timeoutMs);
  if (waitresult == WAIT_OBJECT_0) {
    result = INTERPROC_OK;
    //if (mutexFlags)
      //*mutexFlags = 0;
  }
  else if (waitresult == WAIT_TIMEOUT)
    result = INTERPROC_TIMEOUT;
  else if (waitresult == WAIT_FAILED)
    result = INTERPROC_FAILED;
  else if (waitresult == WAIT_ABANDONED) {
    result = INTERPROC_OK;
    if (mutexFlags)
      *mutexFlags = MUTEXFLAGS_ABANDONED;
  }
  else
    lineabort();
  //logMsg << "Locked mutex" << std::endl;
  return result;
#else

  int waitresult;
  if (timeoutMs < 0)
    waitresult = pthread_mutex_lock(&mutex);
  else {
    timespec ts;
    makeTimeoutMillis(&ts, timeoutMs);
    waitresult = pthread_mutex_timedlock(&mutex, &ts);
    if (waitresult == ETIMEDOUT)
      return INTERPROC_TIMEOUT;
  }

  if (waitresult == EOWNERDEAD) {
    int consistentResult = pthread_mutex_consistent(&mutex);
    logMsg << logValue(consistentResult) << std::endl;
    //lineabort();
    result = INTERPROC_OK;
    if (mutexFlags)
      *mutexFlags = MUTEXFLAGS_ABANDONED;
  }
  else if (waitresult) {
    logMsg << logValue(waitresult) << ' ' << ErrnoToMacroName(waitresult) << std::endl;
    result = INTERPROC_FAILED;
  }
  else {
    result = INTERPROC_OK;
  }
  return result;
#endif
}

void Mutex::unlock() {
  //logMsg << "Unlocking mutex. " << logValue(&mutex) << ' ' << logValue(getpid()) << std::endl;
#if defined _WIN2 || defined __CYGWIN__ || defined __MINGW32__
  int result = ReleaseMutex(mutex);
#else
  int result = pthread_mutex_unlock(&mutex);
#endif
  //logMsg << "Unlocked mutex. " << logValue(result) << std::endl;
}

Lock::Lock(Mutex &mutex, int timeoutMs): mutex(&mutex), locked(1) {
  mutex.lock(timeoutMs);
}

void Lock::unlock() {
  if (locked) {
    mutex->unlock();
    locked = 0;
  }
}

Lock::~Lock() {
  if (locked)
    mutex->unlock();
}

void CountingMutex::operator = (Mutex &mutex) {
  this->mutex = &mutex;
  counter = 0;
}

int CountingMutex::lock(int timeoutMs, int *mutexFlags /*= 0*/) {
  if (counter) {
    counter++;
    return INTERPROC_OK;
  }
  if (!mutex) {
    lineabort();
  }
  int result = mutex->lock(timeoutMs, mutexFlags);
//#if defined _WIN2 || defined __CYGWIN__ || defined __MINGW32__
  //if (result == WAIT_OBJECT_0 || result == WAIT_ABANDONED)
  if (result == INTERPROC_OK)
    counter = 1;
//#else
  //lineabort();
//#endif
  return result;
}

void CountingMutex::unlock() {
  counter--;
  if (!counter)
    mutex->unlock();
  else if (counter < 0)
    logMsg << logValue(counter) << " this is bad" << std::endl;
}

GlobalShare *globalShare = 0;

void Event::initialize() {
#if defined _WIN2 || defined __CYGWIN__ || defined __MINGW32__
  event = CreateEvent(&sa, 0, 0, 0);
#else
  event = eventfd(0, EFD_SEMAPHORE | EFD_NONBLOCK); // Adding EFD_NONBLOCK to prevent more hangs
#endif
}

int Event::setEvent() {
#if defined _WIN2 || defined __CYGWIN__ || defined __MINGW32__
  return SetEvent(event);
#else
  long long buf = 1;
  int result = write(event, &buf, 8);
  if (result != 8) {
    int error = errno;
    logMsg << logValue(result) << ' ' << logValue(error) << ErrnoToMacroName(error) << std::endl;
    lineabort();
  }
  return result;
#endif
}

int Event::wait(int timeoutMs) {
  int result;
#if defined _WIN2 || defined __CYGWIN__ || defined __MINGW32__
  unsigned waitresult = WaitForSingleObject(event, timeoutMs);
  if (waitresult == WAIT_OBJECT_0)
    result = INTERPROC_OK;
  else if (waitresult == WAIT_TIMEOUT)
    result = INTERPROC_TIMEOUT;
  else if (waitresult == WAIT_FAILED)
    result = INTERPROC_FAILED;
  else {
    lineabort();
  }
  return result;
#else
  long long buf;

  int waitresult;
  if (timeoutMs >= 0) {
    fd_set fdset;
    FD_ZERO(&fdset);
    FD_SET(event, &fdset);
    timeval timeout;
    timeout.tv_sec = timeoutMs / 1000;
    timeout.tv_usec = timeoutMs % 1000;
    int selectresult = select(event + 1, &fdset, 0, 0, &timeout);
    if (selectresult == 0) {
      //logMsg << "Hey! Timeout." << std::endl;
      return INTERPROC_TIMEOUT;
    }
    //logMsg << logValue(selectresult) << " so reading " << logValue(event) << std::endl;
  }
  waitresult = read(event, &buf, 8);

  if (waitresult == 8)
    result = INTERPROC_OK;
  else if (waitresult == -1) {
    result = INTERPROC_FAILED;
    int error = errno;
    if (error == EAGAIN) {
      // the event is set to EFD_NONBLOCK so this can occur. If we waited and got nothing,
      // that is effectively a timeout. This happens when the read occurs on another
      // process. It it wasn't set for NONBLOCK, it would cause a hang.
      if (timeoutMs > 0)
        return INTERPROC_TIMEOUT;
    }
    //logMsg << logValue(errno) << ' ' << ErrnoToMacroName(errno) << std::endl;
    //lineabort();
  }
  else {
    lineabort();
  }
  return result;
#endif
}

