
// dep.cpp
// Copyright 2022 Matthew Rickard
// This file is part of dep

// dep is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <https://www.gnu.org/licenses/>.

#include "precomp.h"
#include "dep.h"
#include "logMsg.h"
#include <algorithm>
#include <sys/stat.h>
#include <iomanip>
#include "AbstractProcess.h"
#include "MacroNames.h"
#include <fstream>
#include "matches.h"
#include "Utils.h"
#include <signal.h>
#include <unistd.h>
#include <string.h>
#include "stringify.h"
#if defined __linux__ || defined __CYGWIN__
#include <sys/wait.h>
#include <sys/resource.h>
#include <sys/sysinfo.h>
#include <sys/mman.h>
#endif
#include "clockwork.h"
#include "stream_vector.h"
#include "argv.h"
#include "quote.h"
#include "interproc.h"
#include "DepTree.h"

void Rule::setFolder(const std::string &s) {
  ruleFolder = s;
}

int operator < (const struct timespec &a, const struct timespec &b) {
  return a.tv_sec < b.tv_sec || a.tv_sec == b.tv_sec && a.tv_nsec < b.tv_nsec;
}

int operator >= (const struct timespec &a, const struct timespec &b) {
  return !(a < b);
}

int operator > (const struct timespec &a, const struct timespec &b) {
  return a.tv_sec > b.tv_sec || a.tv_sec == b.tv_sec && a.tv_nsec > b.tv_nsec;
}

std::ostream & operator << (std::ostream &os, const struct timeval &tv) {
  return os << tv.tv_sec << '.' << std::setfill('0') << std::setw(6) << tv.tv_usec;
}

std::ostream & operator << (std::ostream &os, const struct tm &tm) {
  return os << formatDatetime("yy-mm-dd hh:mm:ss",
    tm.tm_year + 1900,
    tm.tm_mon + 1,
    tm.tm_mday,
    tm.tm_hour,
    tm.tm_min,
    tm.tm_sec,
    "", // tm.tm_zone,
    "");
}

std::ostream & operator << (std::ostream &os, const struct timespec &ts) {
#ifdef __MINGW32__
#define LOCALTIME_R_AVAILABLE 0
#else
#define LOCALTIME_R_AVAILABLE 1
#endif
#if LOCALTIME_R_AVAILABLE
  struct tm tm;
  localtime_r(&ts.tv_sec, &tm);
  os << tm;
#else
  struct tm *tm = localtime(&ts.tv_sec);
  os << *tm;
#endif
  int len = 9;
  int ns = ts.tv_nsec;
  while (ns && !(ns % 10)) {
    ns /= 10;
    len--;
  }
  if (ns)
    os << '.' << std::setfill('0') << std::setw(len) << ns;

  std::ostringstream offset;
  int ofs = tm.tm_gmtoff % 60;
  int ofm = (tm.tm_gmtoff / 60) % 60;
  int ofh = tm.tm_gmtoff / 3600;
  if (ofs >= 0)
    os << '+';
  os << std::setfill('0') << std::setw(2) << ofh;
  if (ofm || ofs) {
    os << std::setw(2) << ofm;
    if (ofs)
      os << std::setw(2) << ofs;
  }

  return os;
}

#define caseReturn(a) case a: return #a;

const char *buildStateToMacroName(int buildstate) {
  switch (buildstate) {
  caseReturn(BUILDSTATE_UNKNOWN);
  caseReturn(BUILDSTATE_SOURCES_STALE);
  caseReturn(BUILDSTATE_STALE);
  caseReturn(BUILDSTATE_TOUCH);
  caseReturn(BUILDSTATE_BUILDING);
  caseReturn(BUILDSTATE_STAT_DIFF);
  caseReturn(BUILDSTATE_REBUILT);
  caseReturn(BUILDSTATE_OK);
  caseReturn(BUILDSTATE_REBUILT_SAME);
  caseReturn(BUILDSTATE_NO_SOURCES);
  default: return "Unknown BUILDSTATE";
  }
}

int BuildFile::getFileState(const std::string &folderList, int mode) {
  short fileState;
  if (mode == FOLDERLIST_NONE) {
    //if (fileflag & FILEFLAG_ZBUILDSTATE_VALID) {
      //logMsg << "Returning " << logValue(fileStateNone) << std::endl;
      //return fileStateNone; // crashes and burns
    //}
  }
  else if (mode == FOLDERLIST_SPECIFIC) {
    if (includeVectorMap.count(folderList)) {
      if (includeVectorMap[folderList]->includeVectorflag & FILEFLAG_BUILDSTATE_VALID) {
        if (ruleset->getVerbosity(VERBOSITY_TRACE_STATE, filename)) {
          //*ruleset->outputFile->getOStream() << "Returning cached file build state for "
          //  << logValue(folderList) << " of " << filename << ": "
          //  << buildStateToMacroName(includeVectorMap[folderList]->includeVectorState) << '\n';
        }
        if (includeVectorMap[folderList]->includeVectorState == BUILDSTATE_UNKNOWN) {
          logMsg << "Can't return BUILDSTATE_UNKNOWN" << std::endl;
        }
        else {
          // This caching measurably helps (slightly)
          return includeVectorMap[folderList]->includeVectorState;
        }
      }
    }
    else {
      //logMsg << "Can't retrieve " << logValue(folderList) << ' ' << logValue(filename) << std::endl;
      //lineabort();
    }
  }
  else {
    logMsg << logValue(mode) << std::endl;
    lineabort();
  }

  //if (ruleset->getVerbosity(VERBOSITY_TRACE_STATE, filename)) {
    //*ruleset->outputFile->getOStream() << "Calculating file build state of " << filename << '\n';
  //}

  if (fileflag & FILEFLAG_REBUILT_SAME) {
    //logMsg << "Setting fileState of " << filename << " to BUILDSTATE_REBUILT_SAME\n";
    fileState = BUILDSTATE_REBUILT_SAME;
  }
  else
    fileState = BUILDSTATE_OK;

  if (writeRule) {
    if (fileflag & FILEFLAG_CMD_CHANGED) {
      fileState = BUILDSTATE_STALE;
    }
    fileState = std::min(fileState, (short)writeRule->getRuleState(0));
    if ((fileState == BUILDSTATE_REBUILT) && (fileflag & FILEFLAG_UNCHANGED)) {
      //logMsg << "Setting fileState of " << filename << " to BUILDSTATE_REBUILT_SAME\n";
      fileState = BUILDSTATE_REBUILT_SAME;
    }
  }
  else {
    if (fileflag & FILEFLAG_STAT_CHANGED) {
      // Adding this affects DepTest14. It should not affect DepTest14a (or maybe vice versa)
      //logMsg << "Setting " << filename << " filestate to BUILDSTATE_STAT_DIFF" << std::endl;
      fileState = BUILDSTATE_STAT_DIFF;
    }
  }
  if (gfsRecurseGuard) {
    logMsg << "Recursion at " << filename << std::endl;
    lineabort();
  }
  else {
    gfsRecurseGuard = 1;
    if (mode == FOLDERLIST_SPECIFIC) {
      if (includeVectorMap.count(folderList)) {
        for (auto j = includeVectorMap[folderList]->includeVector.begin();
            j != includeVectorMap[folderList]->includeVector.end() && fileState; j++) {
          //logMsg << filename << " recursing to " << (*j)->filename << '\n';
          short tmpFilestate = (short)(*j)->getFileState(folderList, FOLDERLIST_SPECIFIC);
          if (tmpFilestate == BUILDSTATE_REBUILT_SAME && !(fileflag & FILEFLAG_COMPARE)) {
            // The new thing might be something like this:
            //tmpFilestate = BUILDSTATE_OK;
          }
          fileState = std::min(fileState, tmpFilestate);
          // ZZZ
          if (fileState <= BUILDSTATE_SOURCES_STALE)
            break;
        }
        if (fileState == BUILDSTATE_UNKNOWN)
          lineabort();
        includeVectorMap[folderList]->includeVectorflag |= FILEFLAG_BUILDSTATE_VALID;
        includeVectorMap[folderList]->includeVectorState = fileState;
      }
      else {
        if (includeVectorMap.size() && folderList.size()) {
          logMsg << "Could not find " << logValue(filename) << ' ' << logValue(folderList) << std::endl;
          logMsg << logValue(includeVectorMap.size()) << std::endl;
          for (std::map<std::string, IncludeVector *>::iterator i = includeVectorMap.begin();
              i != includeVectorMap.end(); i++) {
            logMsg << logValue(i->first) << std::endl;
          }
          //logMsg << "This may be unrealistic test data" << std::endl;
          //lineabort();
        }
      }
    }
    else if (mode == FOLDERLIST_NONE);
    else {
      logMsg << logValue(mode) << std::endl;
      lineabort();
    }
    gfsRecurseGuard = 0;
    fileflag |= FILEFLAG_BUILDSTATE_VALID;
  }
  //if (fileState == BUILDSTATE_REBUILT_SAME && !(fileflag & FILEFLAG_COMPARE)) {
    // This is normal. the fileState might come from an include
    //logMsg << filename << " somehow is BUILDSTATE_REBUILT_SAME" << std::endl;
  //}
  if (ruleset->getVerbosity(VERBOSITY_TRACE_STATE, filename)) {
    *ruleset->outputFile->getOStream() << "File build state of "
      << filename << " is " << buildStateToMacroName(fileState) << '\n';
  }

  if (mode == FOLDERLIST_NONE) {
    fileStateNone = fileState;
    //logMsg << "Have set " << logValue(fileStateNone) << std::endl;
    fileflag |= FILEFLAG_BUILDSTATE_VALID;
  }
  /*if (fileState == BUILDSTATE_UNKNOWN) {
    *ruleset->outputFile->getOStream() << "File build state of "
      << filename << " is " << buildStateToMacroName(fileState) << '\n';
  }*/

  return fileState;
}

/*
Determine a rule's (and it's targets') build state.
Calculate it from its sources build states, and if necessary, its sources and targets
stat() outputs including errors, file size and modification time.
*/

int Rule::getRuleState(int debug) {
  if (ruleflag & RULEFLAG_BUILDSTATE_VALID
   || buildstate == BUILDSTATE_REBUILT
   || buildstate == BUILDSTATE_REBUILT_SAME) {
    if (!debug) {
      return buildstate;
    }
  }

  //if (ruleset->getVerbosity(VERBOSITY_TRACE_STATE, this)) {
    //*ruleset->outputFile->getOStream() << "Getting rule build state of " << ruleName() << '\n';
  //}

  ruleflag |= RULEFLAG_BUILDSTATE_VALID;

  int minSourceState = BUILDSTATE_NO_SOURCES;
  for (auto i = sourceVector.begin(); i != sourceVector.end()
      && minSourceState > BUILDSTATE_SOURCES_STALE; i++) {
    if ((*i)->fileflag & FILEFLAG_TENTATIVE)
      continue;
    //if (ruleset->getVerbosity(VERBOSITY_TRACE_STATE, this)) {
      //*ruleset->outputFile->getOStream() << "B Calling getFileState on " << (*i)->filename << '\n';
    //}
    int sourceState = (*i)->getFileState(folderList, FOLDERLIST_SPECIFIC);
    if (debug && sourceState != BUILDSTATE_OK
     || ruleset->getVerbosity(VERBOSITY_TRACE_STATE, this)) {
      *ruleset->outputFile->getOStream() << "     source " << (*i)->filename << " is "
        << buildStateToMacroName(sourceState) << std::endl;
    }
    if (sourceState == BUILDSTATE_REBUILT && (*i)->isExistenceDep())
      //sourceState = BUILDSTATE_REBUILT_SAME;
      // Existence dependency? It is BUILDSTATE_OK
      sourceState = BUILDSTATE_OK;

    minSourceState = std::min(minSourceState, sourceState);
  }

  if ((debug || ruleset->getVerbosity(VERBOSITY_TRACE_STATE, this)) && targetVector.size()
      && minSourceState != BUILDSTATE_SOURCES_STALE) {
    *ruleset->outputFile->getOStream() << targetVector.front()->filename
       << " minSourceState is " << buildStateToMacroName(minSourceState) << std::endl;
  }

  int newbuildstate = BUILDSTATE_SOURCES_STALE;
  int cmdChanged = 0;

  if (minSourceState == BUILDSTATE_SOURCES_STALE
   || minSourceState == BUILDSTATE_BUILDING
   || minSourceState == BUILDSTATE_STALE) {
    newbuildstate = BUILDSTATE_SOURCES_STALE;
  }
  else if (minSourceState == BUILDSTATE_REBUILT) {
    newbuildstate = BUILDSTATE_STALE;
  }
  else if (minSourceState == BUILDSTATE_STAT_DIFF) {
    newbuildstate = BUILDSTATE_STALE;
  }
  else if (ruleset->senseNewer
        && (minSourceState == BUILDSTATE_OK || minSourceState == BUILDSTATE_REBUILT_SAME)
      || minSourceState == BUILDSTATE_NO_SOURCES) {
    if (targetVector.size() && (targetVector[0]->fileflag & FILEFLAG_ALIAS))
      newbuildstate = BUILDSTATE_OK;
    else {
      struct timespec oldestTarget = {0, 0};
      struct timespec newestSource = {1, 0};
      struct timespec newestRebuiltSameSource = {1, 0};
      std::string oldestTargetName, newestSourceName, newestRebuiltSameSourceName;
      int oldestTargetValid = 0;
      int newestSourceValid = 0;
      int newestRebuiltSameSourceValid = 0;
      unsigned n = std::max(sourceVector.size(), targetVector.size());
      for (unsigned i = 0; i < n; i++) {
        if (i < targetVector.size()) {
          if (targetVector[i]->fileflag & FILEFLAG_CMD_CHANGED) {
            newbuildstate = BUILDSTATE_STALE;
            cmdChanged = 1;
            break;
          }
          if (targetVector[i]->fileflag & FILEFLAG_STAT_CHANGED) {
            //if (ruleset->outputFile->getOStream() && ruleset->buildStateVeyrbosity >= 4)
              //*ruleset->outputFile->getOStream() << ruleName() <<
                //" FILEFLAG_STAT_CHANGED; setting newbuildstate to BUILDSTATE_STALE and breaking"
                //<< std::endl;
            // This new code is not sensed but does mirror the FILEFLAG_CMD_CHANGED code above
            //newbuildstate = BUILDSTATE_STALE;
            //break;
          }
          if (targetVector[i]->fileflag & FILEFLAG_PSEUDOFILE) {
            // Don't check the timestamps of pseudofiles. They don't exist
            oldestTargetValid = 0;
            break;
          }
          else if (targetVector[i]->fileflag & FILEFLAG_ALIAS) {
            // Don't check the timestamps of aliases. They don't exist. But don't break either
            oldestTargetValid = 0;
          }
          else {
            targetVector[i]->calculateLocalModTime();
            if (targetVector[i]->modTimeResult) {
              if (ruleset->getVerbosity(VERBOSITY_TRACE_STATE, this))
                if (targetVector[i]->modTimeResult == EINVAL)
                  *ruleset->outputFile->getOStream()
                    << targetVector[i]->filename << " failed validity check" << std::endl;
                else
                  *ruleset->outputFile->getOStream() << "Couldn't get time of target "
                    << targetVector[i]->filename << std::endl;
              oldestTargetValid = 0;
              break;
            }
            else {
              timespec ts = targetVector[i]->localModTime;
              if (targetVector[i]->rebuiltSameTime > ts)
                ts = targetVector[i]->rebuiltSameTime;
              if (!i || ts < oldestTarget) {
                oldestTarget = ts;
                oldestTargetName = targetVector[i]->filename;
                oldestTargetValid = 1;
              }
              if (newestSourceValid && newestSource > targetVector[i]->localModTime) {
                // We need a full rebuild for sure
                break;
              }
              /*if (!i || targetVector[i]->rebuiltSameTime < oldestRebuiltSameTarget) {
                oldestRebuiltSameTarget = targetVector[i]->rebuiltSameTime;
                oldestRebuildSameTargetName = targetVector[i]->filename;
                oldestRebuiltSameTargetValid = 1;
              }*/
            }
          }
        }
        else if (!oldestTargetValid) {
          break;
        }
        if (i < sourceVector.size()) {
          if (sourceVector[i]->fileflag & FILEFLAG_PSEUDOFILE
           || sourceVector[i]->fileflag & FILEFLAG_TENTATIVE
           || sourceVector[i]->fileflag & FILEFLAG_ALIAS
           || sourceVector[i]->isExistenceDep()
          ) {
            // Don't check the timestamps of certain sources (pseudofiles and the like)
          }
          else {
            int fileState = sourceVector[i]->getFileState(folderList, FOLDERLIST_SPECIFIC);
            // removing this condition causes Tutorial 6e to fail
            // keeping this condition causes DogFoodTest3/3a to fail FIXED
            //static int keepCondition = getenvint("KEEPCONDITION", 1);
            if (/*keepCondition &&*/ fileState == BUILDSTATE_REBUILT_SAME
              || fileState == BUILDSTATE_REBUILT && sourceVector[i]->isExistenceDep()) {
            }
            else {
              //logMsg << "Calling calculateIncludeModTime on " << sourceVector[i]->filename << std::endl;
              // YYY Where we use the rebuiltSameTime
              timespec imt, rst;
              sourceVector[i]->calculateIncludeModTime(folderList, imt, rst);
              //if (rst.tv_sec) {
                //logMsg << "Lookee! " << logValue(imt) << ' ' << logValue(rst) << std::endl;
                //lineabort();
              //}
              if (sourceVector[i]->modTimeResult) {
                //logMsg << logValue(sourceVector[i]->modTimeResult) << std::endl;
                if (ruleset->getVerbosity(VERBOSITY_TRACE_STATE, this))
                  *ruleset->outputFile->getOStream() << "Couldn't get time of source "
                    << sourceVector[i]->filename << std::endl;
              }
              else {
                //logMsg << logValue(imt) << std::endl;
                if (!i || imt > newestSource) {
                  newestSource = imt;
                  newestSourceName = sourceVector[i]->filename;
                  newestSourceValid = 1;
                }
                if (!i || rst > newestRebuiltSameSource) {
                  newestRebuiltSameSource = rst;
                  newestRebuiltSameSourceName = sourceVector[i]->filename;
                  newestRebuiltSameSourceValid = 1;
                  //logMsg << ruleName() << ' ' << logValue(newestRebuiltSameSource)
                    //<< std::endl;
                }
                if (oldestTargetValid && imt > oldestTarget) {
                  break;
                }
              }
            }
          }
        }
        /*else if (!newestSourceValid) {
          // If there aren't any source files, there is no newest
          // source file and there won't be a datestamp comparison. Same goes for target files.
          // UPDATE: May still need to call stat and get file existence
          break;
        }*/
      }

      if (oldestTargetValid) {
        if (newestSourceValid)
          if (newestSource > oldestTarget) {
            newbuildstate = BUILDSTATE_STALE;
            //logMsg << "Stale A" << std::endl;
            if (ruleset->getVerbosity(VERBOSITY_TRACE_STATE, this))
              *ruleset->outputFile->getOStream() << "Setting A build state of "
                << ruleName() << " is "
                << buildStateToMacroName(newbuildstate) << '\n'
                << logValue(newestSource) << ' ' << newestSourceName << '\n'
                << logValue(oldestTarget) << ' ' << oldestTargetName << '\n';
          }
          else {
            if (ruleset->optionRebuildAll || (ruleflag & RULEFLAG_REBUILD)) {
              newbuildstate = BUILDSTATE_STALE;
              //logMsg << "Stale B" << std::endl;
            }
            else {
              //logMsg << "Not so stale" << std::endl;
              if (newestRebuiltSameSourceValid && newestRebuiltSameSource > oldestTarget) {
                newbuildstate = BUILDSTATE_TOUCH;
              }
              else {
                newbuildstate = BUILDSTATE_OK;
              }
            }
            if (ruleset->getVerbosity(VERBOSITY_TRACE_STATE, this))
              *ruleset->outputFile->getOStream() << "Setting B build state of "
                << ruleName() << " is " << logValue(newbuildstate) << ' '
                << buildStateToMacroName(newbuildstate) << '\n'
                << logValue(newestSource) << ' ' << newestSourceName << '\n'
                << logValue(oldestTarget) << ' ' << oldestTargetName << '\n';
          }
        else {
          if (ruleset->optionRebuildAll || (ruleflag & RULEFLAG_REBUILD))
            newbuildstate = BUILDSTATE_STALE;
          else {
            //newbuildstate = BUILDSTATE_OK;
            // TODO: Consider copying this block of logic into the two setting blocks above
            if (minSourceState == BUILDSTATE_REBUILT_SAME && !cmdChanged) {
              //logMsg << "Setting C0 buildstate of " << ruleName() << " to BUILDSTATE_TOUCH\n";
              //logMsg << logValue(newbuildstate) << ' ' << buildStateToMacroName(newbuildstate)
                //<< std::endl;
              newbuildstate = BUILDSTATE_TOUCH; // ZZZ
            }
            else
              newbuildstate = BUILDSTATE_OK;
          }
          if (ruleset->getVerbosity(VERBOSITY_TRACE_STATE, this))
            *ruleset->outputFile->getOStream() << "Setting C build state of "
              << ruleName() << " is "
              << buildStateToMacroName(newbuildstate) << '\n'
              //<< logValue(newestSource) << ' ' << newestSourceName << '\n'
              << logValue(oldestTarget) << ' ' << oldestTargetName << '\n';
        }
      }
      else {
        if (minSourceState == BUILDSTATE_REBUILT_SAME && !cmdChanged) {
          //std::cerr << "D: Setting buildstate of " << ruleName() << " to BUILDSTATE_TOUCH\n";
          //std::cerr << logValue(newbuildstate) << ' ' << buildStateToMacroName(newbuildstate) << std::endl;
          newbuildstate = BUILDSTATE_TOUCH; // ZZZ
        }
        else {
          //logMsg << "Setting E buildstate of " << ruleName() << " to BUILDSTATE_STALE\n";
          //logMsg << logValue(minSourceState) << ' ' << buildStateToMacroName(minSourceState) << std::endl;
          newbuildstate = BUILDSTATE_STALE;
        }
      }
    }
  }
  else if (minSourceState == BUILDSTATE_OK) {
    newbuildstate = BUILDSTATE_OK;
  }
  else if (minSourceState == BUILDSTATE_TOUCH) {
    //logMsg << "Setting E buildstate of " << ruleName() << " to BUILDSTATE_TOUCH\n";
    newbuildstate = BUILDSTATE_TOUCH;
  }
  else {
    logMsg << logValue(minSourceState) << ' ' << buildStateToMacroName(minSourceState)
      << std::endl;
    //newbuildstate = BUILDSTATE_OK;
    // minSourceState hasn't been fully identified yet
    //lineabort();
  }
  setState(newbuildstate);
  //logMsg << logValue(targetVector.size()) << std::endl;
  if (ruleset->outputFile && ruleset->outputFile->getOStream()
      && ruleset->getVerbosity(VERBOSITY_TRACE_STATE, this)) {
    *ruleset->outputFile->getOStream() << "Rule build state of ";
    *ruleset->outputFile->getOStream() << ruleName();
    *ruleset->outputFile->getOStream() << " is " /*<< logValue(newbuildstate)
      << ' '*/ << buildStateToMacroName(newbuildstate) << std::endl;
  }
  return buildstate;
}

std::string Rule::ruleName() const {
  if (targetVector.size())
    return targetVector[0]->filename;
  if (commandVector.size())
    return commandVector[0];
  return std::string(pos->filename) + ":" + stringify(pos->line);
}

void BuildFile::calculateLocalModTime() {
  if (fileflag & FILEFLAG_PSEUDOFILE) {
    return;
  }
  if (!(fileflag & FILEFLAG_LOCAL_MODTIME_VALID)) {
    if (fileflag & FILEFLAG_ALIAS) {
      modTimeResult = 0;
      fileSize = 0;
      for (auto k = writeRule->sourceVector.begin(); k != writeRule->sourceVector.end(); k++) {
        if (!(*k)->isExistenceDep()) {
          (*k)->calculateLocalModTime();
          if (k == writeRule->sourceVector.begin() || (*k)->localModTime > localModTime)
            localModTime = (*k)->localModTime;
          if (k == writeRule->sourceVector.begin() || (*k)->modTimeResult)
            modTimeResult = (*k)->modTimeResult;
          fileSize += (*k)->fileSize;
        }
      }
      //logMsg << filename << ' ' << logValue(localModTime) << ' ' << logValue(modTimeResult) << std::endl;
    }
    else {
      struct stat statbuf;
      //logMsg << "Calling stat on " << filename << ' ' << logValue(ruleset->verbosity) << std::endl;
      int returnval = stat(filename.c_str(), &statbuf);
      if (returnval) {
        modTimeResult = errno;
        if (ruleset->getVerbosity(VERBOSITY_STAT, filename)) {
          *ruleset->outputFile->getOStream() // << "calculateLocalModTime called stat on "
            << filename << " stat() returned " << ErrnoToMacroName(modTimeResult) << '\n';
        }
      }
      else {
        modTimeResult = 0;
        if (ruleset->getVerbosity(VERBOSITY_STAT, filename)) {
          *ruleset->outputFile->getOStream() << // "calculateLocalModTime called stat on " <<
            filename << " modtime=" << statbuf.st_mtim << " size=" << statbuf.st_size << '\n';
        }
        // YYY

        if (fileStateNone == BUILDSTATE_REBUILT_SAME) {
          //logMsg << "Calculating rebuiltSameTime" << std::endl;
          rebuiltSameTime = statbuf.st_mtim;
          if (!localModTime.tv_sec)
            localModTime = rebuiltSameTime;
        }
        else {
            if (statbuf.st_mtim > rebuiltSameTime) {
              localModTime = statbuf.st_mtim;
            }
        }

        if (S_ISDIR(statbuf.st_mode)) // The size of directories is somehow 0 on Windows
                                      // and nonzero on Linux. Zero it out permanently
          fileSize = 0;
        else
          fileSize = statbuf.st_size;
        //logMsg << logValue(filename) << ' ' << logValue(fileSize) << std::endl;
  #define ENABLE_NZ_VALIDITY_CHECK 1
  #if ENABLE_NZ_VALIDITY_CHECK
        // Very simple validity check: ignore zero length files
        // Except directories, they are fine (and 0 length on Cygwin)
        if (!S_ISDIR(statbuf.st_mode)) {
          int toosmall = 0;
          if (endsWith(filename.c_str(), ".a")) {
            //logMsg << logValue(filename) << " ends with .a" << std::endl;
            toosmall = 8;
          }
          else {
            //logMsg << logValue(filename) << " does not end with .a" << std::endl;
          }
          if (fileSize <= toosmall && writeRule && !(fileflag & FILEFLAG_ZEROLENGTH)) {
            if (ruleset->getVerbosity(VERBOSITY_NORMAL, filename))
              *ruleset->outputFile->getOStream() << filename
                << " is " << fileSize << " bytes, marking as EINVAL" << std::endl;
            modTimeResult = EINVAL; // Set it to completely invalid
          }
        }
  #endif
      }
    }
    if (modTimeResult) {
      if (modTimeResult == ENOENT || modTimeResult == EINVAL) {
        if (!writeRule && !(fileflag & FILEFLAG_TENTATIVE)) {
          int foundIt = 0;
          for (std::vector<Rule *>::iterator i = readRuleVector.begin();
              i != readRuleVector.end(); i++) {
            Rule *rule = *i;
            if (rule->ruleflag & RULEFLAG_MARKED) {
              //*ruleset->outputFile->getOStream() << "(marked rule)" << std::endl;
              for (std::vector<BuildFile *>::iterator j = rule->targetVector.begin();
                  j != rule->targetVector.end(); j++) {
                BuildFile *bf2 = *j;
                *ruleset->outputFile->getOStream() << filename
                  << " (needed for " << bf2->filename
                  << ") does not exist and has no write rule" << std::endl;
                foundIt = 1;
                ruleset->failed = 1;
                break;
              }
            }
            if (foundIt)
              break;
          }
        }
      }
      else
        *ruleset->outputFile->getOStream() << filename << " error "
          << ErrnoToMacroName(modTimeResult) << std::endl;
    }
    fileflag |= FILEFLAG_LOCAL_MODTIME_VALID;
    /*if (ruleset->getVerbosity(VERBOSITY_STAT, filename)) {
      *ruleset->outputFile->getOStream() << "calculateLocalModTime finished on " <<
        filename << ", localModtime=" << localModTime << '\n';
    }*/
  }
}

void BuildFile::calculateIncludeModTime(const std::string &folderList,
    timespec &imt, timespec &rst) {
  //logMsg << filename << ' ' << logValue(rebuiltSameTime) << std::endl;
  calculateLocalModTime();
  // There are two mod times. One is the local mod time of this file exclusively. It is used
  // to set BUILDSTATE_STAT_DIFF. The other is the include mod time. It considers all
  // included files and is used to look for sources that are newer than objects.

  // The other other is the rebuiltSameTime of this file exclusively, and the other other other
  // is the rebuildSameTime of this file and all included files

  imt = localModTime;
  rst = rebuiltSameTime;
  //logMsg << filename << ' ' << logValue(imt) << std::endl;
  //if (rst.tv_sec)
    //logMsg << filename << ' ' << logValue(rst) << std::endl;

  // FILEFLAG_INCLUDE_MODTIME_VALID is a relatively minor
  // optimization that doesn't work due to the folderList.

  //if (ruleset->getVerbosity(VERBOSITY_STAT, filename)) {
    //*ruleset->outputFile->getOStream() << "calculateIncludeModTime started on " <<
      //filename << '\n';
  //}
  if (1 || !(fileflag & FILEFLAG_INCLUDE_MODTIME_VALID)) {
    if (cmtRecurseGuard) {
      logMsg << "Recursion at " << filename << std::endl;
      lineabort();
    }
    else {
      cmtRecurseGuard = 1;
      fileflag |= FILEFLAG_INCLUDE_MODTIME_VALID;
      if (includeVectorMap.count(folderList)) {
        for (auto j = includeVectorMap[folderList]->includeVector.begin();
            j != includeVectorMap[folderList]->includeVector.end() && getFileState(folderList,
            FOLDERLIST_SPECIFIC); j++) {
          timespec imt2, rst2;
          (*j)->calculateIncludeModTime(folderList, imt2, rst2);
          if (!(*j)->modTimeResult) {
            if (imt2 > imt)
              imt = imt2;
            if (rst2 > rst)
              rst = rst2;
          }
        }
      }
      else {
        if (includeVectorMap.size()) {
          for (std::map<std::string, IncludeVector *>::iterator i = includeVectorMap.begin();
              i != includeVectorMap.end(); i++) {
          }
        }
      }
      cmtRecurseGuard = 0;
    }
  }
  if (ruleset->getVerbosity(VERBOSITY_STAT, filename)) {
    *ruleset->outputFile->getOStream() << // "calculateIncludeModTime finished on " <<
      filename << " includeModtime=" << imt << '\n';
    //if (rst > imt)
      //*ruleset->outputFile->getOStream() << "also rst = " << rst << '\n';
  }
  // YYY Calculate rst too
}

int pipelines = 0;

#ifndef CLOCK_MONOTONIC_RAW
#define CLOCK_MONOTONIC_RAW CLOCK_MONOTONIC
#endif

static void makeMegaPattern(Pattern &pattern, std::string patternList) {
  // todo: unify DONE
  for (;;) {
    size_t pos = patternList.find(',');
    if (pos == std::string::npos)
      break;
    patternList[pos] = '|';
  }
  pattern.setup(patternList.c_str(), std::regex_constants::ECMAScript);
}

static bool skip(const char *name) {
  //static const char *envvar(getenv("SKIP")); // Apparently this isn't good enough
  static int skipInitialized = 0;
  static Pattern pattern("");
  if (!skipInitialized) {
    const char *tmpskip = getenv("SKIP");
    if (tmpskip)
      makeMegaPattern(pattern, tmpskip);
    skipInitialized = 1;
  }
  if (pattern.matches(name))
    return 1;
  return 0;
}

int Rule::startRule(int &resortFlag) {
  if (ruleRunState) {
    logMsg << "Whoops, starting rule " << ruleName() << " that has already been started." << std::endl;
    // What to do:
    // 1. abort? Probably the best thing but leaving for now
    // 2: Repeating with the old RuleRunState? <--
    // 3: Discarding and re-creating the RuleRunState?
    //lineabort();
  }
  //else
    ruleRunState = new RuleRunState();

  if (skip(ruleName().c_str())) {
    logMsg << "Skipping " << ruleName() << '\n';
    markRebuilt(BUILDSTATE_REBUILT);
    return 0;
  }

  clock_gettime(CLOCK_MONOTONIC_RAW, &ruleRunState->startTime);
  if (runlock) {
    ruleset->lockMap[runlock] = this;
    if (!strcmp(runlock, "gui")) {
      logMsg << logValue(globalShare) << std::endl;
      globalShare->mutex[0].lock(-1);
    }
  }
  if (ruleset->optionDryrun) {
    for (auto i: commandVector)
      *ruleset->outputFile->getOStream() << i << std::endl;
    markRebuilt(BUILDSTATE_REBUILT);
    return 0;
  }

  for (auto target = targetVector.begin(); target != targetVector.end(); target++)
    if ((*target)->fileflag & FILEFLAG_COMPARE)
      ReadWholeFile((*target)->filename, ruleRunState->backup[(*target)->filename]);

  if (commandVector.size()) {
    //logMsg << "Setting startedSomething" << std::endl;
    ruleset->startedSomething = 1;
    ruleRunState->runCount = -1;
    //logMsg << "Calling nextCommand\n";
    if (ruleset->getVerbosity(VERBOSITY_PIPELINES, (Rule *)0)) {
      *ruleset->outputFile->getOStream() << "startRule calling nextCommand" << std::endl;
    }
    int result = nextCommand(resortFlag);
    if (result || resortFlag)
      return result;
    buildstate = BUILDSTATE_BUILDING;
  }
  else
    markRebuilt(BUILDSTATE_REBUILT);
  return 0;
}

int Rule::nextCommand(int &resortFlag) {
  repeat:
  //logMsg << "Hello from nextCommand" << std::endl;
  ruleRunState->runCount++;
  if (ruleRunState->runCount >= commandVector.size()) {
    int result = markRebuilt(BUILDSTATE_REBUILT);
    if (result)
      resortFlag = 1;
    delete ruleRunState;
    ruleRunState = 0;
    return 0;
  }
  std::string cmd = commandVector[ruleRunState->runCount];
  //logMsg << logValue(cmd) << std::endl;
  if (cmd.size() && cmd[0] == '@')
    cmd = cmd.erase(0, 1);
  else {
    if (ruleFolder.size()) {
      chdir(ruleFolder.c_str());
      std::cerr << "(in " << ruleFolder << ") ";
    }
    std::cerr << cmd << std::endl;
  }

  ignoreExitCode = 0;
  if (cmd.size()) {
    //if (cmd.starts_with("mkdir ")) { // Jumping the gun on newest libraries
    if (startsWith(cmd.c_str(), "mkdir ")) {
      ignoreExitCode = 1;
    }
    else if (cmd[0] == '-') {
      ignoreExitCode = 1;
      cmd = cmd.erase(0, 1);
    }
  }

  ruleRunState->pipeline.clear();

  if (!strncmp(cmd.c_str(), "export ", 7)) {
    cmd.erase(0, 7);
    //cmd = stripLeadingWhitespace(cmd);
    std::string::size_type equal = cmd.find('=');
    std::string name = cmd.substr(0, equal);
    std::string value = cmd.substr(equal + 1);
    name = stripWhitespace(stripLeadingWhitespace(name));
    value = stripWhitespace(stripLeadingWhitespace(value));
    ruleRunState->pipeline.envmap[name] = value;
    goto repeat;
  }

  //logMsg << logValue(cmd) << std::endl;
  int result = System(cmd, ruleset->outputFile->getFd(O_RDWR | O_CREAT | O_TRUNC),
    -1, 0, SYSTEMFLAG_ASYNC, &ruleRunState->pipeline);

  if (!result) {
    pipelines++;
    if (ruleset->getVerbosity(VERBOSITY_PIPELINES, (Rule *)0)) {
      *ruleset->outputFile->getOStream() << "pipelines increased to " << pipelines << std::endl;
    }
  }
  if (ruleFolder.size())
#if FCHDIR_AVAILABLE
    fchdir(getInitialDir());
#else
    chdir(getInitialDir());
#endif
  return result;
}

int Rule::markRebuilt(int buildState) {
  timespec endTime;
  clock_gettime(CLOCK_MONOTONIC_RAW, &endTime);
  int resortFlag = 0;

  setState(buildState); // should be either BUILDSTATE_REBUILT or BUILDSTATE_REBUILT_SAME

  if (runlock) {
    ruleset->lockMap.erase(runlock);
    if (!strcmp(runlock, "gui")) {
      globalShare->mutex[0].unlock();
    }
  }
  for (auto target = targetVector.begin(); target != targetVector.end(); target++) {
    (*target)->fileflag &= ~FILEFLAG_CMD_CHANGED;
    (*target)->fileflag &= ~FILEFLAG_STAT_CHANGED;
    if (ruleRunState) {
      (*target)->buildSeconds = endTime.tv_sec - ruleRunState->startTime.tv_sec
                              + (endTime.tv_nsec - ruleRunState->startTime.tv_nsec) / 1000000000.0;
    }
    else {
      //logMsg << "Hey! " << ruleName() << " has no ruleRunState" << std::endl;
    }
    if ((*target)->fileflag & FILEFLAG_COMPARE) {
      std::string wholeFile;
      ReadWholeFile((*target)->filename, wholeFile);
      if (ruleRunState && ruleRunState->backup[(*target)->filename] == wholeFile) {
        *ruleset->outputFile->getOStream() << (*target)->filename << " did not change" << std::endl;
        (*target)->fileflag |= FILEFLAG_UNCHANGED;
      }
      else {
        *ruleset->outputFile->getOStream() << (*target)->filename << " changed" << std::endl;
      }
    }
    if ((*target)->includesData) {
      std::fstream is;
      (*target)->fileflag &= ~FILEFLAG_FOUND_INCLUDES;

      for (std::map<std::string, IncludeVector *>::iterator i = (*target)->includeVectorMap.begin();
          i != (*target)->includeVectorMap.end(); i++) {
        // Reconstruct the includes
        // This is OK without the includeDirList, but where would you get it from otherwise?
        std::string foundFile = ruleset->findFile((*target)->includesData->filename,
          /*includeDirList*/0, is, "", ruleFolder.c_str());
        //logMsg << "Calling combinedAddIncludes\n";
        combinedAddIncludes(*target, 0, (*target)->includesData->filename.c_str(),
          &i->second->folders, is, (*target)->includesData->workingDir.c_str(), ruleset->cacheset);
      }
      (*target)->markFile(folderList, 0);
      resortFlag = 1;
    }
  }
  if (ruleRunState)
    ruleRunState->backup.clear();
  return resortFlag;
}

void Rule::setState(int newbuildstate) {
  ruleflag |= RULEFLAG_BUILDSTATE_VALID;
  if (newbuildstate != buildstate) {
    if (ruleset->getVerbosity(VERBOSITY_STATE_CHANGE, this) && targetVector.size())
      if (buildstate == BUILDSTATE_UNKNOWN)
        *ruleset->outputFile->getOStream()
          << targetVector.front()->filename
          << " initially is " << buildStateToMacroName(newbuildstate)
          << '\n';
      else
        *ruleset->outputFile->getOStream()
          << targetVector.front()->filename
          << " changed from " << buildStateToMacroName(buildstate)
          << " to " << buildStateToMacroName(newbuildstate)
          << '\n';
    buildstate = newbuildstate;
    for (auto i = targetVector.begin(); i != targetVector.end(); i++)
      (*i)->invalidateState();
  }
}

void BuildFile::invalidateState() {
  if (fileflag & FILEFLAG_BUILDSTATE_VALID) {
    fileflag &= ~FILEFLAG_BUILDSTATE_VALID;
    for (auto j = readRuleVector.begin(); j != readRuleVector.end(); j++)
      (*j)->invalidateState();

    for (std::map<std::string, IncludeVector *>::iterator i = includeVectorMap.begin();
        i != includeVectorMap.end(); i++) {
      i->second->includeVectorflag &= ~FILEFLAG_BUILDSTATE_VALID;
      for (auto j = i->second->includedByVector.begin(); j != i->second->includedByVector.end(); j++)
        (*j)->invalidateState();
    }
  }
}

void Rule::invalidateState() {
  if (ruleflag & RULEFLAG_BUILDSTATE_VALID) {
    ruleflag &= ~RULEFLAG_BUILDSTATE_VALID;
    for (auto i = targetVector.begin(); i != targetVector.end(); i++)
      (*i)->invalidateState();
  }
}

void Rule::setRunLock(const char *lock) {
  ruleset->lockSet.insert(lock);
  runlock = ruleset->lockSet.find(lock)->c_str();
}

std::string stripCommand(std::string command) {
  command = stripLeadingWhitespace(stripWhitespace(command));
  for (unsigned i = 1; i + 1 < command.size(); i++) {
    if (iswspace(command[i])) {
      command[i] = ' ';
      while (i + 1 < command.size() && iswspace(command[i + 1])) {
        //logMsg << "Changing: " << command << std::endl;
        command.erase(i + 1, 1);
        //logMsg << "Changed to " << command << std::endl;
      }
    }
    else if (command[i] == '\'' || command[i] == '"') {
      int delimiter = command[i];
      for (i++; command[i] != delimiter; i++); // Jump over the quoted string
    }
  }
  return command;
}

void Rule::addCommand(std::string command) {
  size_t pos = command.find("%in ");
  if (pos != std::string::npos)
    command.resize(pos);
  pos = command.find("%out ");
  if (pos != std::string::npos)
    command.resize(pos);
  pos = command.find("%cmpout ");
  if (pos != std::string::npos)
    command.resize(pos);
  pos = command.find("%pseudo ");
  if (pos != std::string::npos)
    command.resize(pos);
  pos = command.find("%exists ");
  if (pos != std::string::npos)
    command.resize(pos);
  pos = command.find("%outz ");
  if (pos != std::string::npos)
    command.resize(pos);

  pos = command.find("%lock ");
  if (pos != std::string::npos)
    command.resize(pos);
  pos = command.find("%priority");
  if (pos != std::string::npos)
    command.resize(pos);
  pos = command.find("%alias ");
  if (pos != std::string::npos)
    command.resize(pos);

  //logMsg << "adding command {" << logValue(command) << "} to rule " << logValue(this) << std::endl;
  if (command.size()) {
    commandVector.push_back(stripCommand(command));
  }
}

std::string normalizePath(std::string a) {
  std::string old = a;
  for (;;) {
    size_t doubleSlash = a.find("//");
    if (doubleSlash == std::string::npos)
      break;
    a.erase(doubleSlash, 1);
  }
  while (a.size() >= 2 && a[0] == '.' && a[1] == '/')
    a.erase(0, 2);
  for (;;) {
    size_t dot = a.find("/./");
    if (dot == std::string::npos)
      break;
    a.erase(dot, 2);
  }
  std::string prefix;
  while (a.size() > 2 && a.substr(0, 3) == "../") {
    prefix += "../";
    a.erase(0, 3);
  }
  for (;;) {
    size_t doubleDot = a.find("/../");
    if (doubleDot == std::string::npos)
      break;
    ssize_t previousSlash = a.rfind('/', doubleDot - 1);
    if (previousSlash == -1) {
      a.erase(0, doubleDot + 4);
    }
    else {
      a.erase(previousSlash, doubleDot + 3 - previousSlash);
    }
  }
  //if (a != old)
    //logMsg << "Changed from " << old << " to " << a << std::endl;
  return prefix + a;
}

std::string joinFolderAndFile(const char *folder, const std::string &file) {
  if (!folder || !*folder)
    return file;
  if (file.size() && file[0] == '/')
    return file;
  std::string result = folder;
  result.append("/");
  result.append(file);
  //logMsg << logValue(result) << std::endl;
  result = normalizePath(result);
  //logMsg << logValue(result) << std::endl;
  return result;
}

std::string joinFolderAndFile(const std::string &folder, const std::string &file) {
  if (!folder.size())
    return file;
  if (file.size() && file[0] == '/')
    return file;
  std::string result = folder + "/" + file;
  //logMsg << logValue(result) << std::endl;
  result = normalizePath(result);
  //logMsg << logValue(result) << std::endl;
  return result;
}

RuleSet::RuleSet() {

  optionTreeIncludes = 1;
  optionRuleIncludes = 0;

  optionDryrun = 0;

  optionRebuildMarked = 0;
  optionRebuildAll = 0;

  optionLink = 1;
  optionMkdir = 1;

  optionFixedOrder = 0;

  senseNewer = 1;
  senseModTime = 1;
  senseSize = 1;
  senseRule = 1;

  maxPipelines = -13;
  maxNonIncludes = -13;
  outputFile = 0;
  failed = 0;
  ignoreFail = 0;
  confirmRuleChanges = 0;

  cacheset = 0;
  modTime.tv_sec = 0;
  modTime.tv_nsec = 0;

  startedSomething = 0;

  for (int i = 0; i < VERBOSITY_ARRAY_SIZE; i++)
    verbosityArray[i] = 0;
  verbosityArray[VERBOSITY_NORMAL] = 1;

  checkBackReferences = 0;
  readCache = 1;
  writeCache = 1;
}

static sighandler_t previousHandler[_NSIG];

#if defined __CYGWIN__
#define USE_MEMFD_CREATE 0
#else
// Not using this on Linux for the moment. Don't remember why
#define USE_MEMFD_CREATE 0
#endif

#if !USE_MEMFD_CREATE
static std::string shmname;

void depExitHandler() {
  //logMsg << "Unlinking " << shmname << std::endl;
  shm_unlink(shmname.c_str());
}

void depSignalHandler(int sig) {
  static int triedThis(0);
  triedThis++;
  if (triedThis > 2) {
    _exit(4);
  }
  if (triedThis > 1) {
    logMsg << olib::argv[0] << " caught signal " << SignalToMacroName(sig)
      << ". Must have totally mangled the signal handlers. _exit()ing this time" << std::endl;
    _exit(4);
  }
  safeSignal(sig, previousHandler[sig], SA_RESTART | SA_NODEFER);
  logMsg << "dep caught signal " << SignalToMacroName(sig)
    << ". restored the old signal handler (" << previousHandler[sig]
    << ") and unlinking shared memory" << std::endl;

  depExitHandler();

  logMsg << "dep unlinked shared memory and re-raising the signal. This should exit the process"
    << std::endl;
  raise(sig); // this shouldn't return
  logMsg << olib::argv[0] << " Hmm. raise() returned without exiting the process. Must have totally "
    "mangled the signal handlers. _exit()ing this time" << std::endl;
  _exit(4);
}
#endif

void RuleSet::makeRulesFromPatterns(int argc, const char *argv[]) {
  unsigned savedRuleVectorSize = ruleVector.size();
  // Convert patterns to targets for BuildFiles that have no write rules
  for (std::map<std::string, BuildFile *>::iterator i = fileMap.begin(); i != fileMap.end(); i++) {
    if (!i->second->writeRule) {
      makeRuleFromPatterns(i->second->filename);
    }
  }
  // Same for command-line targets
  for (int i = 0; i < argc; i++) {
    auto itr = fileMap.find(argv[i]);
    if (itr == fileMap.end()) {
      //logMsg << "Trying to make a rule for " << argv[i] << std::endl;
      makeRuleFromPatterns(argv[i]);
    }
  }
  // Don't change the last rule, it is the default rule. Swap it back
  if (savedRuleVectorSize != ruleVector.size()) {
    std::swap(ruleVector[ruleVector.size() - 1], ruleVector[savedRuleVectorSize - 1]);
  }

  if (getVerbosity(VERBOSITY_SHOW_RULES, (char *)0)) {
    //logMsg << "Showing rules" << std::endl;
    outputTo(*outputFile->getOStream(), 0);
    //logMsg << "Showed rules" << std::endl;
    for (auto i = ruleVector.begin(); i != ruleVector.end(); i++)
      (*i)->ruleflag &= ~RULEFLAG_BUILDSTATE_VALID;
    for (auto i = fileMap.begin(); i != fileMap.end(); i++)
      i->second->fileflag &= ~FILEFLAG_BUILDSTATE_VALID;
  }
  else {
    //logMsg << "Not showing rules" << std::endl;
  }

  if (checkBackReferences)
    checkGraph(*outputFile->getOStream());
}

void RuleSet::makeRuleFromPatterns(const std::string &substitute) {
  for (std::map<std::string, PatternFile *>::iterator j = patternMap.begin();
      j != patternMap.end(); j++) {
    int pos, len;
    //logMsg << "Considering " << substitute
      //<< ' ' << j->second->buildFile.filename << '\n';
    if (j->second->pattern.matches(substitute, &pos, &len)) {
      //logMsg << "Match found " << substitute << ' '
        //<< j->second->buildFile.filename << '\n';

      if (endsWith(substitute.c_str(), ".a")
       || endsWith(substitute.c_str(), ".so")
       || endsWith(substitute.c_str(), ".dll")) {
        //logMsg << "Disregarding " logValue(substitute) << std::endl;
      }

      else if (j->second->buildFile.writeRule) {
        //logMsg << "Has write rule, copying" << std::endl;
        makeRuleFromPattern(j->second->buildFile.writeRule, substitute.substr(pos, len));
      }
      //else {
        //logMsg << "Has no write rule, not copying" << std::endl;
      //}
    }
    //else {
      //logMsg << "That's not a match." << std::endl;
    //}
  }
}

void RuleSet::makeRuleFromPattern(Rule *patternRule, const std::string &substitute) {
  //logMsg << logValue(i->second->filename) << std::endl;
  //logMsg << logValue(j->second->buildFile.filename) << std::endl;
  //logMsg << logValue(substitute) << std::endl;

  // The new way is to construct a new rule string and parse it.
  // The old way was to copy and modify the data. Tedious and complicated and never worked right.

  std::ostringstream oss;

  //patternRule->outputTo(oss, 1);
  patternRule->outputRuleTo(oss, RULEOUTPUT_RESCAN); // Maybe want the 0 for the full rule, it might work better

  // 1 check-tutorial FAIL AT 4G ("../../quux.o does not exist and has no write rule")
  // 1 check-acceptance PASS
  // 0 check-tutorial PASS
  // 0 check-acceptance FAIL AT 2L ("all targets are up to date")

  // Is this due to missing includes? Yes.

  // 1:
  // %out foo.o %in foo.c bar.h
  // %out bar.o %in bar.c bar.h
  // vs 0:
  // %out foo.o %in foo.c
  // %out bar.o %in bar.c

  std::string ruleStr = oss.str();
  //logMsg << "Here is the old pattern rule\n" << logValue(ruleStr) << std::endl;
  size_t pos = std::string::npos;
  for (;;) {
    pos = ruleStr.find('%', pos + 1);
    if (pos == std::string::npos)
      break;
    //logMsg << logValue(pos) << std::endl;
    if (startsWith(&ruleStr[pos], "%out ")
     || startsWith(&ruleStr[pos], "%in ")
     || startsWith(&ruleStr[pos], "%pseudo ")
     || startsWith(&ruleStr[pos], "%rescan")
     || startsWith(&ruleStr[pos], "%rmpout ")
     || startsWith(&ruleStr[pos], "%outz ")
     || startsWith(&ruleStr[pos], "%exists ")
     || startsWith(&ruleStr[pos], "%lock ")
     || startsWith(&ruleStr[pos], "%priority ")
     || startsWith(&ruleStr[pos], "%alias "))
      continue;

    ruleStr.replace(pos, 1, substitute);
    //logMsg << logValue(ruleStr) << std::endl;
    pos += substitute.size() - 1;
  }
  //logMsg << "Here is the new rule: " << logValue(ruleStr) << std::endl;
  depTree->parseString(ruleStr.c_str());
}

void Rule::processIncludes(std::vector<std::string> *includeDirList) {
  //logMsg << "processIncludes started on " << ruleName() << std::endl;
  // --ruleincludes adds extra sources. Use an integer loop
  // variable so as not to invalidate the iterator
  for (unsigned i = 0; i < sourceVector.size(); i++) {
    auto sv = sourceVector[i];

    // note: makeRulesFromPatterns might omit this test
    if (sv->filename.find('%') == std::string::npos) {
      for (std::vector<IncludeRule>::iterator ir = ruleset->includeRuleVector.begin();
          ir != ruleset->includeRuleVector.end(); ir++) {
        for (std::vector<Pattern>::iterator ft = ir->filePatternVector.begin();
            ft != ir->filePatternVector.end(); ft++) {
          Pattern &pattern(*ft);
          int match = pattern.matches(sv->filename);
          if (match) {
            if (ruleset->getVerbosity(VERBOSITY_READ, sv->filename))
              *ruleset->outputFile->getOStream()
                << "Looking in " << sv->filename << std::endl;
            //logMsg << "Looking in " << sv->filename << std::endl;
            std::fstream is;
            std::string foundFile = ruleset->findFile(sv->filename,
              includeDirList, is, "", ruleFolder.c_str());
            //logMsg << "Calling combinedAddIncludes" << std::endl;
            combinedAddIncludes(ruleset->optionTreeIncludes? sv: 0,
              ruleset->optionRuleIncludes? this: 0, foundFile,
              includeDirList, is, ruleFolder.c_str(), ruleset->cacheset);
            //logMsg << "Called combinedAddIncludes" << std::endl;
            break; // Can stop looking now
          }
        }
      }
    }
  }
  folderList = "";
  if (includeDirList)
    for (std::vector<std::string>::iterator i = includeDirList->begin();
        i != includeDirList->end(); i++) {
      if (folderList.size())
        folderList.append(" ");
      folderList.append(*i);
    }
  //logMsg << "processIncludes finished on " << ruleName() << std::endl;
}

int RuleSet::processCmd(int argc, const char *argv[], File *outputFile, Timer &timer) {
  this->outputFile = outputFile;

#ifdef unix
  previousHandler[SIGHUP] = safeSignal(SIGHUP, depSignalHandler, SA_RESTART | SA_NODEFER);
  previousHandler[SIGQUIT] = safeSignal(SIGQUIT, depSignalHandler, SA_RESTART | SA_NODEFER);
  previousHandler[SIGPIPE] = safeSignal(SIGPIPE, depSignalHandler, SA_RESTART | SA_NODEFER);
#endif
  previousHandler[SIGINT] = safeSignal(SIGINT, depSignalHandler, SA_RESTART | SA_NODEFER);
  previousHandler[SIGTERM] = safeSignal(SIGTERM, depSignalHandler, SA_RESTART | SA_NODEFER);

  atexit(depExitHandler);

  makeRulesFromPatterns(argc, argv);

  int foundATarget = 0;
  for (int i = 0; i < argc; i++) {
    std::map<std::string, BuildFile *>::iterator file = fileMap.find(argv[i]);
    //logMsg << logValue(i) << ' ' << logValue(argv[i]) << std::endl;
    if (file == fileMap.end()) {
      std::cerr << "dep: can't find a rule for " << argv[i] << std::endl;
      return -1;
    }
    if (file->second->writeRule) {
      file->second->writeRule->markRule(i);
      if (optionRebuildMarked)
        file->second->writeRule->ruleflag |= RULEFLAG_REBUILD;
    }
    else {
      std::cerr << "dep: can't find a build rule for " << argv[i] << std::endl;
      return -1;
    }
    foundATarget = 1;
  }
  if (argc == 1)
    targetName = std::string(argv[0]) + " is";
  else
    targetName = "all targets are";

  if (!foundATarget) {
    for (std::vector<Rule *>::reverse_iterator i = ruleVector.rbegin(); i != ruleVector.rend(); i++) {
      if (strchr((*i)->ruleName().c_str(), '%')) {
        logMsg << "Skipping " << (*i)->ruleName() << std::endl;
        continue;
      }

      if ((*i)->targetVector.size()) {
        std::cerr << "Targetting " << (*i)->ruleName()
          << std::endl;
      }
      else {
        std::cerr << "(Command \"" << (*i)->ruleName() << "\" has an unknown target)\n";
      }
      (*i)->markRule(0);
      if (optionRebuildMarked)
        (*i)->ruleflag |= RULEFLAG_REBUILD;
      foundATarget = 1;
      break;
    }

    if (!foundATarget) {
      std::cerr << "dep: No rules found and no target selected" << std::endl;
    }
  }

  if (cacheset) {
    //logMsg << "Comparing to cache" << std::endl;
    compareToCache(*cacheset);
    //logMsg << "Compared to cache" << std::endl;
  }

  if (getVerbosity(VERBOSITY_TIMING, (Rule *)0))
    *outputFile->getOStream() << timer.getMicroseconds(TIMER_RESET)
      << " micros comparing cache" << std::endl;

  // Warn about files with no write rule that do not exist
  for (std::map<std::string, BuildFile *>::iterator i = fileMap.begin(); i != fileMap.end(); i++)
    if (!i->second->writeRule)
      i->second->calculateLocalModTime();

  if (getVerbosity(VERBOSITY_TIMING, (Rule *)0))
    *outputFile->getOStream() << timer.getMicroseconds(TIMER_RESET)
      << " micros calculating local mod times" << std::endl;

  /*  This loop has severe performance implications (all$ dep -vtiming says 4.1s on paratux)
   *  The only unit test is -v1-99 DogFoodTest1c which only shows an absence of verbose messages
   *  Removed. New unit tests show that the calculateLocalModTime loop above does all the work
   */

  //for (std::map<std::string, BuildFile *>::iterator i = fileMap.begin(); i != fileMap.end(); i++)
    //if (!i->second->writeRule)
      //i->second->calculateIncludeModTime("", FOLDERLIST_ALL);

  if (getVerbosity(VERBOSITY_TIMING, (Rule *)0))
    *outputFile->getOStream() << timer.getMicroseconds(TIMER_RESET)
      << " micros calculating include mod times" << std::endl;

#define USE_SHARED_MEMORY 1
#if USE_SHARED_MEMORY

#if USE_MEMFD_CREATE
  int shmfd = memfd_create("depshm", 0);
  int error = errno;
  int setenvresult = setenv("DEP_SHM", stringify(shmfd).c_str(), 1);
#else
  shmname = "/dep_" + stringify(getpid());
  // Thanks to Csaba Raduly for pointing out the name must have a leading slash
  int setenvresult = setenv("DEP_SHM", shmname.c_str(), 1);
  int shmfd = shm_open(shmname.c_str(), O_CREAT | O_RDWR /*| O_TRUNC*/, 0644);
  int error = errno;
#endif
  if (shmfd < 0) {
    logMsg << logValue(shmfd) << ' ' << logValue(error) << ' ' << ErrnoToMacroName(error) << std::endl;
    lineabort();
  }
  int tresult = ftruncate(shmfd, 4096);
  globalShare = (GlobalShare *)mmap(0, 4096, PROT_READ | PROT_WRITE, MAP_SHARED, shmfd, 0);
  //logMsg << logValue(setenvresult) << ' ' << logValue(shmfd) << ' ' << logValue(tresult)
    //<< ' ' << logValue(globalShare) << std::endl;
  globalShare->initialize();
  for (int ii = 0; ii < MAX_LOCKS; ii++) {
    globalShare->mutex[ii].initialize();
  }
#endif

  if (getVerbosity(VERBOSITY_TIMING, (Rule *)0))
    *outputFile->getOStream() << timer.getMicroseconds(TIMER_RESET)
      << " micros setting up shared memory" << std::endl;

  return run(outputFile, &timer);
}

static void addSourcesTo(Rule *rule, std::vector<Rule *> &v, std::vector<Rule *> &from) {
  for (auto buildFile = rule->sourceVector.begin();
      buildFile != rule->sourceVector.end(); buildFile++) {
    if ((*buildFile)->writeRule) {
      auto j = v.begin();
      for (; j != v.end(); j++)
        if (*j == (*buildFile)->writeRule) {
          //logMsg << "Breaking because " << *j << ' ' << (*j)->ruleName()
            //<< " is the same as " << (*i)->writeRule << ' ' << (*i)->filename << std::endl;
          break;
        }
      if (j == v.end()) {
        for (auto k = from.begin(); k != from.end(); k++) {
          if ((*buildFile)->writeRule == *k) {
            //logMsg << "Adding " << (*i)->writeRule << ' ' << (*i)->filename
            //<< " as a source of " << rule << ' ' << rule->ruleName()
            //<< std::endl;
            v.push_back((*buildFile)->writeRule);
            addSourcesTo((*buildFile)->writeRule, v, from);
            break;
          }
        }
      }
    }
#define CHECK_INCLUDEFILES 1
#if CHECK_INCLUDEFILES
    // Make sure included files are counted as sources during sorting.
    // Note: can't see any effect in unit tests. Make sure this is unit tested!
    for (auto i = (*buildFile)->includeVectorMap.begin();
        i != (*buildFile)->includeVectorMap.end(); i++) {
      for (auto includeFile = i->second->includeVector.begin();
          includeFile != i->second->includeVector.end(); includeFile++) {
        // This might do one level but it should be recursive
        if ((*includeFile)->writeRule) {
          auto j = v.begin();
          for (; j != v.end(); j++)
            if (*j == (*includeFile)->writeRule) {
              //logMsg << "Breaking because " << *j << ' ' << (*j)->ruleName()
                //<< " is the same as " << (*i)->writeRule << ' ' << (*i)->filename << std::endl;
              break;
            }
          if (j == v.end()) {
            for (auto k = from.begin(); k != from.end(); k++) {
              if ((*includeFile)->writeRule == *k) {
                //logMsg << "Adding " << (*includeFile)->writeRule->ruleName() << ' '
                //<< (*includeFile)->filename << " as a source of " << rule->ruleName()
                //<< std::endl;
                v.push_back((*includeFile)->writeRule);
                addSourcesTo((*includeFile)->writeRule, v, from);
                //lineabort();
                break;
              }
            }
          }
        }
      }
    }
#endif
  }
}

int operator != (const timespec &a, const timespec &b) {
  return a.tv_sec != b.tv_sec || a.tv_nsec != b.tv_nsec;
}

timespec Rule::getFailTime() const {
  if (targetVector.size()) {
    //logMsg << logValue(targetVector[0]->failTime) << std::endl;
    return targetVector[0]->failTime;
  }
  return timespec{0, 0};
}

int whichComesFirst(Rule *a, Rule *b) {
  int result = 0;
  if (a->cmdOrder != b->cmdOrder) {
    if (a->cmdOrder < b->cmdOrder)
      result = -6;
    else
      result = 6;
  }
  else if (a->implementsHeader() != b->implementsHeader()) {
    if (a->implementsHeader())
      result = -5;
    else
      result = 5;
  }
  else if (!a->ruleset->optionFixedOrder && a->getFailTime() != b->getFailTime()) {
    // Targets which have recently failed (up to about 15% of the jobs)
    if (a->getFailTime() > b->getFailTime())
      result = -4;
    else
      result = 4;
  }
  else if (a->priority != b->priority) {
    if (a->priority > b->priority)
      result = -3;
    else
      result = 3;
  }
  else if (!a->runlock != !b->runlock) {
    if (a->runlock)
      result = -2;
    else
      result = 2;
  }
  else if (!a->ruleset->optionFixedOrder
      && a->getCompletionSeconds() != b->getCompletionSeconds()) {
    if (a->getCompletionSeconds() > b->getCompletionSeconds())
      result = -1;
    else
      result = 1;
  }
  return result;
}

double Rule::getCompletionSeconds() {
  if (ruleflag & RULEFLAG_COMPLETIONSECONDS_VALID)
    return completionSeconds;
  ruleflag |= RULEFLAG_COMPLETIONSECONDS_VALID;
  double tmpCompletionSeconds = 0;
  for (auto i = targetVector.begin(); i != targetVector.end(); i++) {
    tmpCompletionSeconds = std::max(tmpCompletionSeconds, (*i)->getCompletionSeconds());
  }
  completionSeconds = tmpCompletionSeconds;
  return tmpCompletionSeconds;
}

double BuildFile::getCompletionSeconds() const {
  double result = 0;
  for (auto i = readRuleVector.begin(); i != readRuleVector.end(); i++) {
    result = std::max(result, (*i)->getCompletionSeconds());
  }
  //logMsg << logValue(buildSeconds) << std::endl;
  result += buildSeconds;
  //logMsg << logValue(result) << ' ' << logValue(buildSeconds) << std::endl;
  return result;
}

int Rule::implementsHeader() {
  if (ruleflag & RULEFLAG_IMPLHDR_VALID)
    return savedImplementsHeader;
  ruleflag |= RULEFLAG_IMPLHDR_VALID;
  for (auto i = sourceVector.begin(); i != sourceVector.end(); i++) {
    if ((*i)->fileflag & FILEFLAG_TENTATIVE)
      continue;
    std::string sourceBase = removeExt(BaseName((*i)->filename.c_str()));
    for (std::map<std::string, IncludeVector *>::iterator jj = (*i)->includeVectorMap.begin();
        jj != (*i)->includeVectorMap.end(); jj++)
      for (auto j = jj->second->includeVector.begin(); j != jj->second->includeVector.end(); j++) {
        std::string includeBase = removeExt(BaseName((*j)->filename.c_str()));
        //logMsg << logValue(sourceBase) << ' ' << logValue(includeBase) << std::endl;
        if (includeBase == sourceBase) {
          for (auto k = targetVector.begin(); k != targetVector.end(); k++) {
            (*j)->calculateLocalModTime();
            (*k)->calculateLocalModTime();
            if ((*j)->localModTime > (*k)->localModTime) {
              savedImplementsHeader = 1;
              return savedImplementsHeader;
            }
          }
          savedImplementsHeader = 0;
          return savedImplementsHeader;
        }
      }
  }
  savedImplementsHeader = 0;
  return savedImplementsHeader;
}

int firstThingsFirst(Rule *a, Rule *b, int &significantDifference) {
  int result = whichComesFirst(a, b);

  if (0) {
    if (result < 0)
      logMsg << a->ruleName() << " comes before " << b->ruleName() << ' ' << logValue(result) << std::endl;
    else if (result > 0)
      logMsg << b->ruleName() << " comes before " << a->ruleName() << ' ' << logValue(result) << std::endl;
    else
      logMsg << a->ruleName() << " is about the same as " << b->ruleName() << ' ' << logValue(result) << std::endl;
  }

  if (result)
    significantDifference = 1;

  return result < 0;
}

void RuleSet::sortRuleVector(std::vector<Rule *> &ruleVector, int findOrder) {
  //logMsg << "Sorting rule vector" << std::endl;
  if (ruleVector.size() < 2) {
    if (ruleVector.size())
      (*ruleVector.begin())->findOrder = findOrder;
    //logMsg << "sortRuleVector: too darn trivial" << std::endl;
    return;
  }

  // That algorithm:
  // 1 choose a rule
  // 2 find all its sources
  // 3 recursively sort all of its sources
  // 4 find all its non-sources
  // 5 output itself
  // 6 recursively sort all its non-sources

  int minState = 99;

  Rule *mostImportant = *ruleVector.begin();

  auto i = ruleVector.begin();
  i++;
  int significantDifference = 0;
  for (; i != ruleVector.end(); i++) {
    if (firstThingsFirst(*i, mostImportant, significantDifference)) {
      mostImportant = *i;
      //significantDifference = 1;
      //logMsg << logValue(mostImportant->ruleName()) << std::endl;
    }
  }
  //logMsg << "Starting with " << mostImportant->ruleName() << '\n' << std::endl;
  mostImportant->findOrder = findOrder;

  // 2 find all its sources
  std::vector<Rule *> sources;
  //logMsg << "Calling addSourcesTo" << std::endl;
  addSourcesTo(mostImportant, sources, ruleVector);

  // 3 recursively sort all of its sources
  sortRuleVector(sources, findOrder + significantDifference);

  // 4 find all its non-sources
  std::vector<Rule *> nonsources;
  for (auto i = ruleVector.begin(); i != ruleVector.end(); i++) {
    if (*i == mostImportant)
      continue;
    auto j = sources.begin();;
    for (; j != sources.end(); j++) {
      if (*j == *i)
        break;
    }
    if (j == sources.end())
      nonsources.push_back(*i);
  }

  if (sources.size() + nonsources.size() + 1 != ruleVector.size()) {
    logMsg << "Oh no! sorted " << ruleVector.size() << " rules into "
      << sources.size() << " + 1 + " << nonsources.size() << std::endl;
    for (auto i = ruleVector.begin(); i != ruleVector.end(); i++)
      logMsg << (*i)->ruleName() << std::endl;
    logMsg << "Sources:" << std::endl;
    for (auto i = sources.begin(); i != sources.end(); i++)
      logMsg << (*i)->ruleName() << std::endl;
    logMsg << "Most important:" << std::endl;
    logMsg << mostImportant->ruleName() << std::endl;
    logMsg << "Non-Sources:" << std::endl;
    for (auto i = nonsources.begin(); i != nonsources.end(); i++)
      logMsg << (*i)->ruleName() << std::endl;
    logMsg << std::endl;
    lineabort();
  }

  // 5 output itself
  sources.push_back(mostImportant);

  // 6 recursively sort all its non-sources
  sortRuleVector(nonsources, findOrder + significantDifference);

  // 7 concatentate and return
  sources.insert(sources.end(), nonsources.begin(), nonsources.end());
  std::swap(sources, ruleVector);
  //logMsg << logValue(sources.size()) << ' ' << logValue(ruleVector.size()) << std::endl;
}

timeval &operator -= (timeval &a, const timeval &b) {
  if (a.tv_usec < b.tv_usec) {
    a.tv_usec += 1000000;
    a.tv_sec -= 1;
  }
  a.tv_sec -= b.tv_sec;
  a.tv_usec -= b.tv_usec;
  return a;
}

timespec getCommonTime() {
  static int initialized = 0;
  static timespec commonTime;
  if (!initialized) {
    clock_gettime(CLOCK_REALTIME, &commonTime);
    initialized = 1;
  }
  return commonTime;
}

int RuleSet::waitForRuleExit(int &resortFlag, int startMore) {
  int wstatus, wstatusfinal = 0;

  fullRestart:

#ifdef __MINGW32__
  int pid = 0;
  lineabort();
#else
  //logMsg << "Waiting" << std::endl;
  int pid = wait(&wstatus);
  //logMsg << "Wait complete " << logValue(pid) << ' '
    //<< logValue(wstatus) << ' ' << waitString(wstatus) << std::endl;
#endif

  if (pid == -1) {
    int err = errno;
    if (err == ECHILD) {
      //logMsg << "Checkpoint: returning 0" << std::endl;
      if (pipelines > 0) {
        logMsg << "Reducing pipelines from " << pipelines << " to " << (pipelines - 1) << std::endl;
        pipelines--;
      }
      return wstatusfinal;
    }
    logMsg << "errno = " << err << ' ' << ErrnoToMacroName(err) << std::endl;
    lineabort();
  }

  int foundIt = 0;

  loopRestart:

  // Look through all processes in the current pipeline in all rules
  for (std::vector<Rule *>::iterator i = ruleVector.begin(); i != ruleVector.end(); i++) {
    //logMsg << "Looking at a rule, " << logValue((*i)->targetVector.size()) << std::endl;
    if ((*i)->ruleRunState) {
      for (std::vector<Process *>::iterator j = (*i)->ruleRunState->pipeline.processVector.begin();
          j != (*i)->ruleRunState->pipeline.processVector.end(); j++) {
        //logMsg << logValue(*j) << std::endl;
        if (*j && pid == (*j)->child) {
          //logMsg << "Found it " << logValue(wstatus) << std::endl;
          foundIt = 1;
          if (wstatus) {
            if ((*i)->ruleFolder.size())
              failedCommands << "(in " << (*i)->ruleFolder << ") ";
            if ((*i)->ignoreExitCode)
              failedCommands << '-';

            std::vector<Process *>::iterator k;
            for (k = (*i)->ruleRunState->pipeline.processVector.begin();
                k != (*i)->ruleRunState->pipeline.processVector.end(); k++) {

              if (k != (*i)->ruleRunState->pipeline.processVector.begin())
                failedCommands << " | ";

              failedCommands << (*k)->args[0];
              for (unsigned i = 1; i < (*k)->args.size(); i++)
                failedCommands << ' ' << (*k)->args[i];
            }
            std::vector<BuildFile *>::iterator kk;
            for (kk = (*i)->targetVector.begin(); kk != (*i)->targetVector.end(); kk++) {
              if ((*kk)->fileflag & FILEFLAG_PSEUDOFILE)
                failedCommands << " [" << (*kk)->filename << ']';
            }

            failedCommands << " (" << waitString(wstatus);
            if ((*i)->ignoreExitCode) {
              failedCommands << " (ignored)";
              std::cerr << "(ignoring " << (*i)->ruleName() << ")\n";
            }
            failedCommands << ")\n";
            //logMsg << logValue(failedCommands) << std::endl;
            //logMsg << logValue((*i)->ignoreExitCode) << std::endl;
            if ((*i)->ignoreExitCode)
              wstatus = 0;
            else {
            }
          }
          (*j)->child = -1;
          if (!wstatusfinal)
            wstatusfinal = wstatus;
          for (auto target = (*i)->targetVector.begin(); target != (*i)->targetVector.end(); target++) {
            if (wstatus) {
              if (WIFEXITED(wstatus) && WEXITSTATUS(wstatus) == 3) {
                // 3 is a common code for "stopping as requested". Don't mark it as failed
              }
              else {
                (*target)->failTime = getCommonTime();
              }
            }
          }

          // An old bug:
          //   Exiting with exitcode 1
          // pipelines reduced to 0
          // waitForRuleExit calling nextCommand
          // Eh? There was a non-zero exit code. Stop!

          // Check if any processes in the pipeline are still running
          for (std::vector<Process *>::iterator k = (*i)->ruleRunState->pipeline.processVector.begin();
              k != (*i)->ruleRunState->pipeline.processVector.end(); k++) {
            if ((*k)->child != -1) {
              //logMsg << "Something is still running" << std::endl;
              //return wstatus;
              //repeat = 1;
              if (getVerbosity(VERBOSITY_WAIT_FOR_RESULT, (Rule *)0)) {
                *outputFile->getOStream() << "[Something is still running]" << std::endl;
              }
              goto fullRestart;
            }
          }

          pipelines--;
          if (getVerbosity(VERBOSITY_PIPELINES, (Rule *)0)) {
            *outputFile->getOStream() << "pipelines reduced to " << pipelines << std::endl;
            // XXX is hgxlex there? Yes it is
            //System("ls -l genlinux64/hgslex.cpp");
            //System("pwd");
            //Sleep(500);
            //System("ls -l genlinux64/hgslex.cpp");
          }

          if (!failed && !wstatus) {
            //logMsg << "Calling nextCommand " << logValue(startMore) << std::endl;
            if (startMore) {
              if (getVerbosity(VERBOSITY_PIPELINES, (Rule *)0)) {
                *outputFile->getOStream() << "waitForRuleExit calling nextCommand" << std::endl;
              }
              int result = (*i)->nextCommand(resortFlag);
              //logMsg << logValue(result) << std::endl;
              if (resortFlag) {
                //logMsg << "Checkpoint: returning " << logValue(wstatus) << std::endl;
                return wstatusfinal;
              }
              if (getVerbosity(VERBOSITY_WAIT_FOR_RESULT, (Rule *)0)) {
                *outputFile->getOStream() << "[Restarting the loops]" << std::endl;
              }
              goto loopRestart; // nextCommand invalidated the iterators, so start from scratch
            }
          }
        }
      }
    }
  }
  if (!foundIt && getVerbosity(VERBOSITY_NORMAL, (Rule *)0)) {
    *outputFile->getOStream() << "[unrecognized child process]" << std::endl;
    logMsg << "Unrecognized child process" << std::endl;
    goto fullRestart;
  }
  //logMsg << "Checkpoint: returning " << logValue(wstatus) << std::endl;
  return wstatusfinal;
}

void RuleSet::interpretVerbosity(String argVerbosity) {
  //logMsg << logValue(argVerbosity.s) << std::endl;
  if (!strcmp(argVerbosity.s, "normal"))
    setVerbosity(VERBOSITY_NORMAL);
  else if (!strcmp(argVerbosity.s, "stat"))
    setVerbosity(VERBOSITY_STAT);
  else if (!strcmp(argVerbosity.s, "rulechange"))
    setVerbosity(VERBOSITY_RULE_CHANGE);
  else if (!strcmp(argVerbosity.s, "read"))
    setVerbosity(VERBOSITY_READ);
  else if (!strcmp(argVerbosity.s, "uptodate"))
    setVerbosity(VERBOSITY_UPTODATE);
  else if (!strcmp(argVerbosity.s, "unselected"))
    setVerbosity(VERBOSITY_UNSELECTED);
  else if (!strcmp(argVerbosity.s, "marked")) // was "rundecision" but does not really do that
    setVerbosity(VERBOSITY_MARKED);
  else if (!strcmp(argVerbosity.s, "statechange"))
    setVerbosity(VERBOSITY_STATE_CHANGE);
  else if (!strcmp(argVerbosity.s, "start"))
    setVerbosity(VERBOSITY_START);
  else if (!strcmp(argVerbosity.s, "waitforresult"))
    setVerbosity(VERBOSITY_WAIT_FOR_RESULT);
  else if (!strcmp(argVerbosity.s, "tracerun"))
    setVerbosity(VERBOSITY_TRACE_RUN);
  else if (!strcmp(argVerbosity.s, "findgenerated"))
    setVerbosity(VERBOSITY_FIND_GENERATED);
  else if (!strcmp(argVerbosity.s, "tracestate"))
    setVerbosity(VERBOSITY_TRACE_STATE);
  else if (!strcmp(argVerbosity.s, "showorder"))
    setVerbosity(VERBOSITY_SHOW_ORDER);
  else if (!strcmp(argVerbosity.s, "showrules"))
    setVerbosity(VERBOSITY_SHOW_RULES);
  else if (!strcmp(argVerbosity.s, "checkgraph"))
    //setVerbosity(VERBOSITY_CHECK_GRAPH);
    checkBackReferences = 1;
  else if (!strcmp(argVerbosity.s, "pipelines"))
    setVerbosity(VERBOSITY_PIPELINES);
  else if (!strcmp(argVerbosity.s, "timing"))
    setVerbosity(VERBOSITY_TIMING);
  else {
    const char *p = argVerbosity.s;
    for (;;) {
      const char *dash = strchr(p, '-');
      if (dash == p)
        dash = strchr(p + 1, '-');
      if (dash) {
        int from = atoi(p);
        int to = atoi(dash + 1);
        for (int i = from; i <= to; i++) {
          setVerbosity(i);
        }
        break;
      }
      else {
        if (*p)
          for (int i = 0; i <= atoi(p); i++) {
            setVerbosity(i);
          }
        else {
          setVerbosity(1);
        }
        break;
      }
    }
  }
}

void RuleSet::setVerbosity(int argVerbosity) {
  if (argVerbosity >= 0 && argVerbosity < VERBOSITY_ARRAY_SIZE) {
    verbosityArray[argVerbosity] = 1;
  }
  else if (argVerbosity == -1) {
    for (int i = 0; i < VERBOSITY_ARRAY_SIZE; i++)
      verbosityArray[i] = 0;
  }
}

int RuleSet::getVerbosity(int argVerbosity, const Rule *rule) const {
  if (rule) {
    for (auto bf = rule->sourceVector.begin(); bf != rule->sourceVector.end(); bf++) {
      if (getVerbosity(argVerbosity, *bf))
        return 1;
    }
    for (auto bf = rule->targetVector.begin(); bf != rule->targetVector.end(); bf++) {
      if (getVerbosity(argVerbosity, *bf))
        return 1;
    }
  }
  if (outputFile && outputFile->getOStream()
      && argVerbosity >= 0 && argVerbosity < VERBOSITY_ARRAY_SIZE)
    return verbosityArray[argVerbosity];
  return 0;
}

int RuleSet::getVerbosity(int argVerbosity, const BuildFile *bf) const {
  return getVerbosity(argVerbosity, bf->filename);
}

int RuleSet::getVerbosity(int argVerbosity, String filename) const {
  //logMsg << logValue(filename) << std::endl;
  //lineabort();
  if (outputFile && outputFile->getOStream()) {
    //if (!safestrcmp(filename.s, tracefile.c_str()))
      //return 1;
    if (filename.s && tracefileSet.count(filename.s))
      return 1;
    if (argVerbosity >= 0 && argVerbosity < VERBOSITY_ARRAY_SIZE)
      return verbosityArray[argVerbosity];
  }
  return 0;
}

int touch(const char *filename) {
  // touch all the targets by calling futimens()
  int fd = open(filename, O_WRONLY | O_CREAT, 0666);
  if (fd == -1) {
    int error = errno;
    logMsg << ErrnoToMacroName(error) << " opening " << filename << '\n';
  }
  else {
    //logMsg << "Touching " << filename << '\n';
    struct timespec times[2];
    times[0].tv_nsec = UTIME_OMIT;
    times[1].tv_nsec = UTIME_NOW;
    int result = futimens(fd, times);
    if (result) {
      int error = errno;
      logMsg << ErrnoToMacroName(error) << " calling futimens on " << filename << '\n';
    }
    close(fd);
  }
  return 0;
}

int RuleSet::run(File *outputFile, Timer *timer) {

  pipelines = 0;
  if (failed)
    return -1;

  this->outputFile = outputFile;
  getInitialDir();

  flattenOldFailTimes();

#define RULESET_RUN_SERIAL    0
#define RULESET_RUN_PARALLEL  1
#define RULESET_RUN_ALGORITHM 1

#if RULESET_RUN_ALGORITHM == RULESET_RUN_PARALLEL
  static int staticMaxPipelines = getenvint("PROCESSORS", get_nprocs());
  if (maxPipelines < 1) {
    maxPipelines = std::max(staticMaxPipelines, 1);
  }
  int currentMaxPipelines = (maxPipelines + 1) / 2;
  //logMsg << logValue(currentMaxPipelines) << std::endl;
#endif

  resortLabel:
  //logMsg << "Proceeding from resortLabel" << std::endl;

  // Copy out and sort the marked rules
  std::vector<Rule *> newRuleVector;
  for (std::vector<Rule *>::iterator i = ruleVector.begin(); i != ruleVector.end(); i++) {
    if ((*i)->ruleflag & RULEFLAG_MARKED) {
      if ((*i)->buildstate == BUILDSTATE_REBUILT
       || (*i)->buildstate == BUILDSTATE_REBUILT_SAME) {
        if (getVerbosity(VERBOSITY_UNSELECTED, *i)) {
          *outputFile->getOStream() << "Skipping rebuilt " << (*i)->ruleName() << "\n";
        }
      }
      else if ((*i)->buildstate == BUILDSTATE_BUILDING) {
        if (getVerbosity(VERBOSITY_UNSELECTED, *i)) {
          *outputFile->getOStream() << "Skipping in-progress " << (*i)->ruleName() << "\n";
        }
      }
      else {
        //*outputFile->getOStream() << "Adding " << (*i)->ruleName() << "\n";
        newRuleVector.push_back(*i);
      }
    }
    else {
      if (getVerbosity(VERBOSITY_UNSELECTED, *i)) {
        *outputFile->getOStream() << "Skipping unselected " << (*i)->ruleName() << "\n";
      }
    }
  }
  // ZZZ

  if (timer && getVerbosity(VERBOSITY_TIMING, (Rule *)0))
    *outputFile->getOStream() << timer->getMicroseconds(TIMER_RESET)
      << " micros marking and copying rules" << std::endl;

  sortRuleVector(newRuleVector, 0);

  if (timer && getVerbosity(VERBOSITY_TIMING, (Rule *)0))
    *outputFile->getOStream() << timer->getMicroseconds(TIMER_RESET)
      << " micros sorting the rule vector" << std::endl;

  if (getVerbosity(VERBOSITY_SHOW_ORDER, (Rule *)0)) {
    *outputFile->getOStream() << "Marked and sorted rules:\n";
    for (std::vector<Rule *>::iterator i = newRuleVector.begin();
        i != newRuleVector.end(); i++) {
      for (int j = 0; j < (*i)->findOrder; j++)
        *outputFile->getOStream() << ' ';
      *outputFile->getOStream() << (*i)->findOrder << ' ';
      if ((*i)->targetVector.size())
        for (unsigned j = 0; j < (*i)->targetVector.size(); j++) {
          if (j)
            *outputFile->getOStream() << ' ';
          (*i)->targetVector[j]->outputTo(*outputFile->getOStream());
        }
      else
        *outputFile->getOStream() << " (no target)";
      if ((*i)->sourceVector.size()) {
        *outputFile->getOStream() << " %in";
        for (unsigned j = 0; j < (*i)->sourceVector.size(); j++) {
          *outputFile->getOStream() << ' ' << quoteAsNec((*i)->sourceVector[j]->filename);
          if ((*i)->sourceVector[j]->fileflag & FILEFLAG_TENTATIVE)
            *outputFile->getOStream() << "(TENTATIVE)";
          if ((*i)->sourceVector[j]->fileflag & FILEFLAG_EXISTENCE)
            *outputFile->getOStream() << "(EXISTS)";
        }
      }
      else
        *outputFile->getOStream() << " (no sources)";
      if ((*i)->commandVector.size()) {
        *outputFile->getOStream() << " { ";
        for (unsigned j = 0; j < (*i)->commandVector.size(); j++) {
          if (j)
            *outputFile->getOStream() << "; ";
          *outputFile->getOStream() << (*i)->commandVector[j];
        }
        *outputFile->getOStream() << " }\n";
      }
      else
        *outputFile->getOStream() << " (no command)\n";
    }
  }

  startedSomething = 0;

#if RULESET_RUN_ALGORITHM == RULESET_RUN_PARALLEL
  std::vector<Rule *>::iterator startingPoint = newRuleVector.begin();
  //logMsg << "About to enter the startingPoint loop" << std::endl;
  for (;;) {
    //logMsg << "Just inside the startingPoint loop" << std::endl;
    for (;;) {
      if (startingPoint == newRuleVector.end()) {
        //logMsg << "Breaking" << std::endl;
        break;
      }
      int ruleState = (*startingPoint)->getRuleState(0);
      if (ruleState == BUILDSTATE_REBUILT
       || ruleState == BUILDSTATE_REBUILT_SAME
       || ruleState == BUILDSTATE_BUILDING) {
        //logMsg << "Incrementing the starting point" << std::endl;
        startingPoint++;
      }
      else if (ruleState == BUILDSTATE_OK) {
        if (getVerbosity(VERBOSITY_UPTODATE, *startingPoint)) {
          *outputFile->getOStream() << "Skipping up-to-date "
            << (*startingPoint)->ruleName() << '\n';
        }
        startingPoint++;
      }
      else if (ruleState == BUILDSTATE_STALE
            || ruleState == BUILDSTATE_TOUCH
            || ruleState == BUILDSTATE_SOURCES_STALE) {
        break;
      }
      else {
        logMsg << buildStateToMacroName(ruleState) << std::endl;
      }
    }
    std::vector<Rule *>::iterator i = startingPoint;
    //logMsg << "Starting the newRuleVector loop" << std::endl;
    for (; i != newRuleVector.end(); i++) {
      int result = 0;
      int ruleState = (*i)->getRuleState(0);
      if (ruleState == BUILDSTATE_TOUCH) {
        for (std::vector<BuildFile *>::iterator j = (*i)->targetVector.begin();
            j != (*i)->targetVector.end(); j++) {
          if (!((*j)->fileflag & (FILEFLAG_PSEUDOFILE | FILEFLAG_ALIAS))) {
            touch((*j)->filename.c_str());
            *outputFile->getOStream() << "Touched " << (*j)->filename << std::endl;
          }
          else {
            //*outputFile->getOStream() << "Not touching " << (*j)->filename << std::endl;
          }
        }

        (*i)->markRebuilt(BUILDSTATE_REBUILT_SAME); // Orig
      }
      if (ruleState == BUILDSTATE_STALE) {
        if (!(*i)->runlock || !lockMap.count((*i)->runlock)) {
          if (getVerbosity(VERBOSITY_START, *i))
            *outputFile->getOStream() << "Starting " << (*i)->ruleName()
             << ' ' << buildStateToMacroName((*i)->getRuleState(0)) << std::endl;
          int resortFlag = 0;
          result = (*i)->startRule(resortFlag);
          if (result) {
            waitForAllPipelines();
            return result;

          }
          if (resortFlag) {
            logMsg << "Going to resortLabel" << std::endl;
            goto resortLabel;
          }
          break;
        }
      }
      //logMsg << "Looping around" << std::endl;
    }

    if (i == newRuleVector.end()) {
      if (!pipelines) {
        // 7/10/2025
        // This is currently the _only_ way out of the loop. An open question is, is there
        // any work left to do? E.g. mark off any more %pseudos or %aliases with no bodies

        //std::cerr << "Breaking because no pipelines" << std::endl;
        break;
      }
    }
    else if (pipelines >= currentMaxPipelines) {
      if (currentMaxPipelines < maxPipelines) {
        currentMaxPipelines++;
        if (getVerbosity(VERBOSITY_WAIT_FOR_RESULT, *i)) {
          *outputFile->getOStream() << logValue(currentMaxPipelines) << "\n";
        }
      }
    }
    else {
      //logMsg << logValue(pipelines) << ' ' << logValue(currentMaxPipelines) << std::endl;
      continue;
    }
    int resortFlag = 0;
    for (;;) {
      if (getVerbosity(VERBOSITY_WAIT_FOR_RESULT, (Rule *)0)) {
        *outputFile->getOStream() << "[Waiting for results]" << std::endl;
      }
      if (getVerbosity(VERBOSITY_PIPELINES, (Rule *)0)) {
        *outputFile->getOStream() << logValue(pipelines) << ", waiting" << std::endl;
      }
      int result = waitForRuleExit(resortFlag, 1);
      if (getVerbosity(VERBOSITY_PIPELINES, (Rule *)0)) {
        *outputFile->getOStream() << logValue(pipelines) << ", waited" << std::endl;
      }
      //logMsg << logValue(result) << ' ' << logValue(resortFlag) << std::endl;
      if (getVerbosity(VERBOSITY_WAIT_FOR_RESULT, (Rule *)0)) {
        *outputFile->getOStream() << "[Waited for results]" << std::endl;
      }
      if (result) {
        failed = 1;
        if (globalShare && !ignoreFail) {
          //logMsg << "Setting stopnow" << std::endl;
          globalShare->stopnow = 1;
        }
        std::cerr << "Command failed: " << logValue(waitString(result)) << std::endl;
        if (pipelines) {
          waitForAllPipelines();
        }
        std::cerr << "All commands exited" << std::endl;
        std::cerr << "Failed commands were:" << std::endl;
        std::cerr << failedCommands.str();
        if (getVerbosity(VERBOSITY_TRACE_RUN, (Rule *)0))
          *outputFile->getOStream() << "[returning]" << std::endl;
        return result;
      }
      if (pipelines < currentMaxPipelines)
        break;
      if (getVerbosity(VERBOSITY_PIPELINES, (Rule *)0)) {
        *outputFile->getOStream() << logValue(pipelines) << ", repeating wait" << std::endl;
      }
    }

    if (resortFlag) {
      if (getVerbosity(VERBOSITY_TRACE_RUN, (Rule *)0))
        *outputFile->getOStream() << "[going to resortLabel]" << std::endl;
      goto resortLabel;
    }
    //logMsg << "Going around" << std::endl;
  }

#elif RULESET_RUN_ALGORITHM == RULESET_RUN_SERIAL
  for (auto i = newRuleVector.begin(); i != newRuleVector.end(); i++) {
    int result = 0;
    if (ruleset->outputFile && ruleset->outputFile->getOStream()
        && ruleset->verbosity[VERBOSITY_TRACE_STATE]) {
      //*ruleset->outputFile->getOStream() << "D Calling getFileState on "
        //<< sourceVector[i]->filename << '\n';
    }
    if (/*optionRebuildAll ||*/ ((*i)->getFileState() == BUILDSTATE_STALE)) {
      if ((*i)->commandVector.size())
        logMsg << "Running " << (*i)->commandVector[0] << std::endl;
      result = (*i)->start();
      //logMsg << "Setting startedSomething" << std::endl;
      startedSomething = 1;
      result = waitForRuleExit();
    }
    else
      if ((*i)->commandVector.size())
        logMsg << "Not running " << (*i)->commandVector[0] << std::endl;
    if (result)
      return result;
  }
  logMsg << "Outputting details to dep.error" << std::endl;
  std::ofstream ofs("dep.error", std::ios::out | std::ios::trunc);
  outputTo(ofs);
  ofs << "Here is the newRuleVector" << std::endl;
  for (auto i = newRuleVector.begin(); i != newRuleVector.end(); i++) {
    if ((*i)->targetVector.size())
      ofs << logValue((*i)->targetVector.front()->filename) << std::endl;
  }
  ofs << "That was the newRuleVector" << std::endl;
#else
  bork bork bork
#endif
  if (!failed) {
    // There were no rule command failures. Print out anything that did not build
    for (auto i = newRuleVector.begin(); i != newRuleVector.end(); i++) {
      int ruleState = (*i)->getRuleState(0);
      if (ruleState != BUILDSTATE_OK
       && ruleState != BUILDSTATE_REBUILT
       && ruleState != BUILDSTATE_REBUILT_SAME)
      {
        *outputFile->getOStream() << "dep: Could not build " << (*i)->ruleName()
          << " state is " << buildStateToMacroName(ruleState) << std::endl;
        (*i)->getRuleState(1);
        failed = 1;
      }
    }
  }
  if (!startedSomething && !failed) {
    //std::cerr << "dep: all targets are up to date" << std::endl;
    std::cerr << "dep: " + targetName + " up to date" << std::endl;
  }
  return 0;
}

void RuleSet::waitForAllPipelines() {
  while (pipelines) {
    int resortFlag;
    waitForRuleExit(resortFlag, 0);
  }
}

Rule *RuleSet::rulesetAddRule(const Position *pos) {
  Rule *rule = new Rule(this);
  //logMsg << logValue(rule) << std::endl;
  rule->ruleflag = 0;
  rule->pos = pos;
  ruleVector.push_back(rule);
  return rule;
}

RuleSet::~RuleSet() {
  //logMsg << "Deleting RuleSet" << std::endl;
  for (std::vector<Rule *>::iterator i = ruleVector.begin(); i != ruleVector.end(); i++)
    delete *i;
  for (std::map<std::string, BuildFile *>::iterator i = fileMap.begin(); i != fileMap.end(); i++)
    delete i->second;
  //for (std::map<std::string, BuildFile *>::iterator i = patternMap.begin(); i != patternMap.end(); i++)
    //delete i->second;
  for (std::map<std::string, Cmdspec *>::iterator i = syntaxMap.begin(); i != syntaxMap.end(); i++)
    delete i->second;
}

BuildFile *RuleSet::getBuildFile(const char *filename, int flags) {
  if (strchr(filename, '%')) {
    auto itr = patternMap.find(filename);
    if (itr == patternMap.end()) {
      std::string p1(filename);
      ReplaceAll(p1, ".", "\\.");
      ReplaceAll(p1, "%", "(.*)");
      //logMsg << logValue(p1) << std::endl;
      PatternFile *buildfile = new PatternFile(this, p1.c_str());
      buildfile->buildFile.filename = filename;
      buildfile->buildFile.fileflag = flags;
      patternMap[filename] = buildfile;
      return &buildfile->buildFile;
    }
    //logMsg << "Found " << logValue(filename) << std::endl;
    return &itr->second->buildFile;
  }
  else {
    auto itr = fileMap.find(filename);
    if (itr == fileMap.end()) {
      BuildFile *buildfile = new BuildFile(this);
      buildfile->filename = filename;
      buildfile->fileflag = flags;
      fileMap[filename] = buildfile;
      return buildfile;
    }
    //logMsg << "Found " << logValue(filename) << std::endl;
    return itr->second;
  }
}

void Rule::addTarget(const char *filename, int flags, const char *sourceFile, int sourceLine) {
  if (!filename) {
    std::cerr << sourceFile << ':' << sourceLine
      << " addTarget(): no filename supplied, is the syntax line missing a colon?\n\n";
  }
  else
    addTarget(std::string(filename), flags, sourceFile, sourceLine);
}

void Rule::addTarget(const std::string &filename, int flags, const char *sourceFile, int sourceLine) {
  //logMsg << "addTarget started" << std::endl;
  BuildFile *buildfile = ruleset->getBuildFile(joinFolderAndFile(ruleFolder, filename).c_str(), 0);
  buildfile->fileflag |= flags & (FILEFLAG_PSEUDOFILE
    | FILEFLAG_COMPARE | FILEFLAG_ZEROLENGTH | FILEFLAG_ALIAS);
  buildfile->fileflag &= ~FILEFLAG_TENTATIVE;
  if (ruleset->optionMkdir) {
    if (filename.find('/') != std::string::npos) {
      std::string dirname = DirName(filename.c_str());
      if (dirname != "..") {
        addSource(dirname, FILEFLAG_EXISTENCE | FILEFLAG_TENTATIVE);
      }
    }
  }
  //logMsg << logValue(filename) << ' ' << buildfile->writeRule << std::endl;
  if (buildfile->writeRule) {
    if (ruleset->outputFile->getOStream())
      if (sourceLine != buildfile->writeRule->pos->line)
        if (!strcmp(sourceFile, buildfile->writeRule->pos->filename))
          *ruleset->outputFile->getOStream() << sourceFile << ' ' << sourceLine << ": "
            << filename << " already has a rule at " << buildfile->writeRule->pos->filename
            << ' ' << buildfile->writeRule->pos->line << std::endl;
  }
  else {
    buildfile->writeRule = this;
    targetVector.push_back(buildfile);
  }
  //logMsg << "addTarget finished" << std::endl;
}

int Rule::addSource(const std::string &filename, int flags) {
  BuildFile *buildfile = ruleset->getBuildFile(joinFolderAndFile(ruleFolder, filename).c_str(), flags);

  if (flags & FILEFLAG_EXISTENCE)
    buildfile->fileflag |= FILEFLAG_EXISTENCE;

  if (!(flags & FILEFLAG_TENTATIVE)) {
    buildfile->fileflag &= ~FILEFLAG_TENTATIVE;
  }

  std::vector<Rule *>::iterator i = buildfile->readRuleVector.begin();
  for (; i != buildfile->readRuleVector.end(); i++)
    if (*i == this)
      return 0;
  buildfile->readRuleVector.push_back(this);
  sourceVector.push_back(buildfile);
  if (flags & FILEFLAG_TENTATIVE)
    return 0;
  return 1;
}

// Tricky thing about includes is that they SOMETIMES depend on the Build Command.
// Because it has the -I options.

int BuildFile::treeAddInclude(const std::string &filename, const std::string &folderList) {
  BuildFile *buildfile = ruleset->getBuildFile(filename.c_str(), 0);
  if (!includeVectorMap.count(folderList)) {
    // don't think this ever happens
    lineabort();
    //logMsg << "Adding " << folderList << " to includeVectorMap of " << filename << std::endl;
    //includeVectorMap[folderList] = new IncludeVector;
  }
  if (!buildfile->includeVectorMap.count(folderList)) {
    //logMsg << "Adding " << folderList << " to includeVectorMap of " << buildfile->filename << std::endl;

    IncludeVector *newIncludeVector = new IncludeVector;
    buildfile->includeVectorMap[folderList] = newIncludeVector;
    // YYY
    // Extra populating code. This doesn't work well enough though because this section only gets
    // called 4 times. It should get called 6 times. It gets called by combinedAddIncludes

    //logMsg << "Populating newIncludeVector for " << filename
      //<< ", " << logValue(folderList) << std::endl;
    /*
    std::string folderList2;
    while (folderList2.size()) {
      std::string field = GetField(folderList2, " ");
      newIncludeVector->folders.push_back(field);
    }
    */
  }
  for (unsigned i = 0; i < includeVectorMap[folderList]->includeVector.size(); i++)
    if (includeVectorMap[folderList]->includeVector[i] == buildfile) {
      //logMsg << "Not adding " << filename << " to " << this->filename
        //<< ' ' << logValue(folderList) << std::endl;
      return 0;
  }
  //logMsg << "Adding " << filename << " to " << this->filename
    //<< ' ' << logValue(folderList) << std::endl;
  buildfile->includeVectorMap[folderList]->includedByVector.push_back(this);
  includeVectorMap[folderList]->includeVector.push_back(buildfile);

  return 1;
}

void Rule::markRule(int n) {
  if (!(ruleflag & RULEFLAG_MARKED)) {
    if (ruleset->getVerbosity(VERBOSITY_MARKED, ruleName())) {
      *ruleset->outputFile->getOStream() << "Marking rule " << ruleName() << std::endl;
    }
    ruleflag |= RULEFLAG_MARKED;
    for (std::vector<BuildFile *>::iterator j = sourceVector.begin(); j != sourceVector.end(); j++)
      (*j)->markFile(folderList, n);
    cmdOrder = std::min(cmdOrder, (short)n);
  }
}

void BuildFile::markFile(const std::string &folderList, int cmdOrder) {
  //logMsg << logValue(filename) << ' ' << logValue(folderList) << std::endl;
  if (ruleset->getVerbosity(VERBOSITY_MARKED, filename)) {
    *ruleset->outputFile->getOStream() << "Marking file " << filename << std::endl;
  }
  if (writeRule)
    writeRule->markRule(cmdOrder);
  if (mfRecurseGuard) {
    logMsg << "Recursion at " << filename << std::endl;
    logMsg << includeVectorMap.size() << std::endl;
    for (std::map<std::string, IncludeVector *>::iterator i = includeVectorMap.begin();
        i != includeVectorMap.end(); i++)
      for (unsigned j = 0; j < i->second->includeVector.size(); j++)
        logMsg << filename << " includes "
          << logValue(i->second->includeVector[j]->filename) << std::endl;
    lineabort();
  }
  else {
    if (includeVectorMap.count(folderList)) {
      mfRecurseGuard = 1;
      for (std::vector<BuildFile *>::iterator j = includeVectorMap[folderList]->includeVector.begin();
          j != includeVectorMap[folderList]->includeVector.end(); j++) {
        (*j)->markFile(folderList, cmdOrder);
      }
      mfRecurseGuard = 0;
    }
  }
}

Rule::Rule(RuleSet *ruleset): runlock(0), ruleset(ruleset),
  ruleRunState(0),
  priority(0),
  ruleflag(0),
  buildstate(BUILDSTATE_UNKNOWN), cmdOrder(32767), findOrder(0) {
}

PatternFile::PatternFile(RuleSet *ruleset, const char *pattern):
  buildFile(ruleset), pattern(pattern) {
}

BuildFile::BuildFile(RuleSet *ruleset): writeRule(0), ruleset(ruleset), fileflag(0) {
  fileSize = 0;
  localModTime.tv_sec = 0;
  localModTime.tv_nsec = 0;
  failTime.tv_sec = 0;
  failTime.tv_nsec = 0;
  rebuiltSameTime.tv_sec = 0;
  rebuiltSameTime.tv_nsec = 0;
  //logMsg << "Cleared rebuiltSameTime" << std::endl;
  buildSeconds = 0;
  fileStateNone = BUILDSTATE_UNKNOWN;
  mfRecurseGuard = 0;
  cmtRecurseGuard = 0;
  gfsRecurseGuard = 0;
  includesData = 0;
}

IncludeVector::IncludeVector() {
  includeVectorState = BUILDSTATE_UNKNOWN;
  includeVectorflag = 0;
}

void RuleSet::addIncludeSyntax(const std::string &s) {
  const char *space = findUnquoted(s.c_str(), " \t");
  IncludeRule includeRule(s.substr(1, space - s.c_str() - 2).c_str());
  //includeRule.syntax = s.substr(1, space - s.c_str() - 2);
  while (*space) {
    const char *wildcards = skipWhitespace(space);
    std::string wildcard1;
    space = skipAndDequote(wildcards, &wildcard1);
    std::string newPattern;
    if (wildcard1.substr(0, 2) == "*.")
      newPattern = std::string(".*\\.") + wildcard1.substr(2);
    else
      lineabort();
    //logMsg << logValue(wildcard1) << ' ' << logValue(newPattern) << std::endl;
    includeRule.filePatternVector.push_back(Pattern(newPattern.c_str(),
      std::regex_constants::ECMAScript));
  }
  includeRuleVector.push_back(includeRule);
}

std::string RuleSet::findFile(const std::string &filename, const std::vector<std::string> *folders,
    std::fstream &is, const char *reading, const char *workingDir) const {
  int detailed = 0;
  tryagain:
  //if (detailed)
    //logMsg << "Looking for " << logValue(filename) << std::endl;

  is.open(filename);
  if (is.is_open())
    return filename;

  // ^ This stops the message:

  // parallax:testfire$ dep
  // Warning: cannot find path of CppTree.h included by ../plextest/testmain.cpp (workingDir=).
  // If CppTree.h is a generated file, it's generating command should go
  // above anything that reads ../plextest/testmain.cpp
  // .Returning the base name CppTree.h
  // Dir list: gencyg64 ../ignite ../ignite/gencyg64 ../plextest ../plextest/gencyg64 ../ ""

  if (folders) {
    for (auto folder: *folders) {
      std::string longfilename = joinFolderAndFile(joinFolderAndFile(workingDir, folder), filename);
      //if (detailed)
        //logMsg << logValue(longfilename) << std::endl;
      is.open(longfilename);
      if (is.is_open())
        return longfilename;
    }
  }
  // Is it a generated file?
  for (auto i = fileMap.begin(); i != fileMap.end(); i++) {
    if (endsWith(("/" + i->first).c_str(), ("/" + filename).c_str())) {
      if (getVerbosity(VERBOSITY_FIND_GENERATED, i->first))
        *outputFile->getOStream() << "Considering generated file " << i->first << std::endl;
      if (folders)
        for (auto folder: *folders) {
          std::string longfilename = joinFolderAndFile(joinFolderAndFile(workingDir, folder), filename);
          if (longfilename == i->first) {
            if (getVerbosity(VERBOSITY_FIND_GENERATED, i->first))
              *outputFile->getOStream() << "Returning generated file " << i->first << std::endl;
            return i->first;
          }
        }
    }
  }

  // Is it in the same folder as we are reading?
  std::string readingDir = DirName(reading);
  std::string longfilename = joinFolderAndFile(readingDir, filename);
  if (detailed) {
    //logMsg << logValue(reading) << std::endl;
    //logMsg << logValue(DirName(reading)) << std::endl;
    //logMsg << logValue(longfilename) << std::endl;
  }
  is.open(longfilename);
  if (is.is_open())
    return longfilename;

  // Is it in the same folder as we are working?
  longfilename = joinFolderAndFile(workingDir, filename);
  if (detailed) {
    //logMsg << logValue(workingDir) << std::endl;
    //logMsg << logValue(longfilename) << std::endl;
  }
  is.open(longfilename);
  if (is.is_open())
    return longfilename;

  if (!detailed && getVerbosity(VERBOSITY_NORMAL, filename)) {
    *outputFile->getOStream() << "Warning: cannot find path of " << filename;
    if (reading && *reading)
      *outputFile->getOStream() << "\nincluded by " << reading;
    *outputFile->getOStream() << " (" << logValue(workingDir) << ").\n";
#ifdef DONT_BOTHER
    *outputFile->getOStream() << "If " << filename
      << " is a generated file, it's generating command should go above anything that\nreads "
      << reading << ". If it is a system file, it should be in angle brackets.\nReturning the base"
         " name " << filename << std::endl;
#endif
    if (folders && folders->size()) {
      *outputFile->getOStream() << "Dir list:";
      for (auto folder: *folders)
        //if (folder.size())
          *outputFile->getOStream() << ' ' << quoteAsNec(folder);
      *outputFile->getOStream() << /* "End of dir list" << */ std::endl;
    }
    if (!detailed) {
      detailed = 1;
      goto tryagain; // Do it again with extra trace code (currently all of which is commented out)
    }
  }
  return filename;
}

void BuildFile::outputTo(std::ostream &os) {
  if (fileflag & FILEFLAG_PSEUDOFILE)
    os << "%pseudo ";
  else if (fileflag & FILEFLAG_COMPARE)
    os << "%cmpout ";
  else if (fileflag & FILEFLAG_ZEROLENGTH)
    os << "%outz ";
  else if (fileflag & FILEFLAG_ALIAS)
    os << "%alias ";
  else
    os << "%out ";

  os << quoteAsNec(filename);
}

void RuleSet::checkGraph(std::ostream &os) {
  int foundIt = 0;
  for (std::vector<Rule *>::iterator i = ruleVector.begin(); i != ruleVector.end(); i++) {
    for (std::vector<BuildFile *>::iterator j = (*i)->sourceVector.begin();
        j != (*i)->sourceVector.end(); j++) {
      foundIt = 0;
      for (std::vector<Rule *>::iterator k = (*j)->readRuleVector.begin();
          k != (*j)->readRuleVector.end(); k++) {
        if (*k == *i) {
          foundIt = 1;
          break;
        }
      }
      if (foundIt) {
        os << "Read rule " << (*i)->ruleName() << " points back to source file "
          << (*j)->filename << '\n';
      }
      else {
        os << "Read rule " << (*i)->ruleName() << " does not point back to source file "
          << (*j)->filename << std::endl;
        lineabort();
      }
    }
    for (std::vector<BuildFile *>::iterator j = (*i)->targetVector.begin();
        j != (*i)->targetVector.end(); j++) {
      if ((*j)->writeRule == *i) {
        os << "Write rule " << (*i)->ruleName() << " points back to target file "
          << (*j)->filename << '\n';
      }
      else {
        os << "Write rule " << (*i)->ruleName() << " does not point back to target file "
          << (*j)->filename << std::endl;
        lineabort();
      }
    }
  }
  for (std::map<std::string, BuildFile *>::iterator i = fileMap.begin();
      i != fileMap.end(); i++) {
    for (std::vector<Rule *>::iterator j = i->second->readRuleVector.begin();
        j != i->second->readRuleVector.end(); j++) {
      foundIt = 0;
      for (std::vector<BuildFile *>::iterator k = (*j)->sourceVector.begin();
          k != (*j)->sourceVector.end(); k++) {
        if (*k == i->second) {
          foundIt = 1;
          break;
        }
      }
      if (foundIt) {
        os << "Source file " << i->second->filename << " points back to read rule "
          << (*j)->ruleName() << '\n';
      }
      else {
        os << "Source file " << i->second->filename << " does not point back to read rule "
          << (*j)->ruleName() << std::endl;
        lineabort();
      }
    }
    if (i->second->writeRule) {
      foundIt = 0;
      for (std::vector<BuildFile *>::iterator k = i->second->writeRule->targetVector.begin();
          k != i->second->writeRule->targetVector.end(); k++) {
        if (*k == i->second) {
          foundIt = 1;
          break;
        }
      }
      if (foundIt) {
        os << "Target file " << i->second->filename << " points back to write rule "
          << i->second->writeRule->ruleName() << '\n';
      }
      else {
        os << "Target file " << i->second->filename << " does not point back to write rule "
          << i->second->writeRule->ruleName() << std::endl;
        lineabort();
      }
    }
    for (std::map<std::string, IncludeVector *>::iterator j = i->second->includeVectorMap.begin();
        j != i->second->includeVectorMap.end(); j++) {
      for (std::vector<BuildFile *>::iterator jj = j->second->includeVector.begin();
          jj != j->second->includeVector.end(); jj++) {
        foundIt = 0;
        if ((*jj)->includeVectorMap.count(j->first))
          for (std::vector<BuildFile *>::iterator k
              = (*jj)->includeVectorMap[j->first]->includedByVector.begin();
              k != (*jj)->includeVectorMap[j->first]->includedByVector.end(); k++) {
            if (*k == i->second) {
              foundIt = 1;
              break;
            }
          }
        if (foundIt) {
          os << "Including file " << i->second->filename << " points back to included file "
            << (*jj)->filename << '\n';
        }
        else {
          logMsg << "Including file " << i->second->filename << " does not point back to included file "
            << (*jj)->filename << std::endl;
          lineabort();
        }
      }
      for (std::vector<BuildFile *>::iterator jj = j->second->includedByVector.begin();
          jj != j->second->includedByVector.end(); jj++) {
        foundIt = 0;
        if ((*jj)->includeVectorMap.count(j->first))
          for (std::vector<BuildFile *>::iterator k
              = (*jj)->includeVectorMap[j->first]->includeVector.begin();
              k != (*jj)->includeVectorMap[j->first]->includeVector.end(); k++) {
            if (*k == i->second) {
              foundIt = 1;
              break;
            }
          }
        if (foundIt) {
          os << "Included file " << i->second->filename << " points back to including file "
            << (*jj)->filename << '\n';
        }
        else {
          logMsg << "Included file " << i->second->filename << " does not point back to including file "
            << (*jj)->filename << std::endl;
          lineabort();
        }
      }
    }
  }
  os << "The graph checks out\n";
}

void Rule::outputRuleTo(std::ostream &os, int flags) {
  if (!targetVector.size())
    return;
  /*if (flags & RULEOUTPUT_CMDONLY) {
    lineabort();
    // Outputting only the cmds might be silly
    for (auto j: commandVector)
      os << j << '\n';
    return;
  }*/
  int started = 0;
  for (auto j: targetVector) {
    if (started)
      os << ' ';
    started = 1;
    j->outputTo(os);
  }
  int currentType = -1;
  for (auto j: sourceVector) {
    if (j->fileflag & FILEFLAG_TENTATIVE)
      continue;
    if (j->isExistenceDep() && currentType != FILEFLAG_EXISTENCE) {
      os << " %exists";
      currentType = FILEFLAG_EXISTENCE;
    }
    else if (!j->isExistenceDep() && currentType != 0) {
      os << " %in";
      currentType = 0;
    }
    os << ' ' << quoteAsNec(j->filename);
  }
  if (priority) {
    os << " %priority";
  }
  if (runlock) {
    os << " %lock " << runlock;
  }
  os << " {\n";
  if (ruleFolder.size())
    os << "cd " << quoteAsNec(ruleFolder) << "\n";
  if (flags & RULEOUTPUT_RESCAN)
    os << "%rescan\n";
  for (auto j: commandVector)
    os << j << '\n';
  os << "}\n\n";
}

void RuleSet::outputTo(std::ostream &os, int outputComments) {
  os << '\n';
  for (std::map<std::string, Cmdspec *>::iterator i = syntaxMap.begin();
      i != syntaxMap.end(); i++) {
    os << "syntax " << i->first;
    i->second->outputTo(os, outputComments);
    os << '\n';
  }
  for (std::vector<Rule *>::iterator i = ruleVector.begin(); i != ruleVector.end(); i++) {
    (*i)->outputRuleTo(os, 0);
  }

  for (std::map<std::string, BuildFile *>::iterator i = fileMap.begin();
      i != fileMap.end(); i++) {
    if (i->second->fileflag & FILEFLAG_TENTATIVE) {
      if (!i->second->includeVectorMap.size()) {
        continue;
      }
    }
    if (strchr(i->second->filename.c_str(), '%')) {
      continue;
    }
    int filestate = i->second->getFileState("", FOLDERLIST_NONE);
    os << "fileinfo " << quoteAsNec(i->first) << "\n buildstate ";

    switch (filestate)
    {
    case BUILDSTATE_OK: os << "OK"; break;
    case BUILDSTATE_STALE: os << "STALE"; break;
    case BUILDSTATE_SOURCES_STALE: os << "SOURCES_STALE"; break;
    case BUILDSTATE_REBUILT:
      os << "REBUILT";
      i->second->fileflag &= ~FILEFLAG_LOCAL_MODTIME_VALID;
      i->second->calculateLocalModTime();
      break;
    case BUILDSTATE_REBUILT_SAME:
      //logMsg << "Outputting REBUILT_SAME" << std::endl;
      os << "REBUILT_SAME";
      i->second->fileflag &= ~FILEFLAG_LOCAL_MODTIME_VALID;
      i->second->calculateLocalModTime();
      break;
    case BUILDSTATE_STAT_DIFF: os << "STAT_DIFF"; break;
    case BUILDSTATE_BUILDING: os << "BUILDING"; break;
    case BUILDSTATE_TOUCH: os << "TOUCH"; break;
    default:
      os << buildStateToMacroName(i->second->getFileState("", FOLDERLIST_NONE)) << ' '
         << std::flush;
      logMsg << "No mapping for " << i->second->getFileState("", FOLDERLIST_NONE) << ' '
             << buildStateToMacroName(i->second->getFileState("", FOLDERLIST_NONE)) << std::endl;
      lineabort();
    }
    os << "\n changetime ";
    if (i->second->fileflag & FILEFLAG_LOCAL_MODTIME_VALID) {
      if (i->second->modTimeResult) {
        os << ErrnoToMacroName(i->second->modTimeResult);
        if (i->second->modTimeResult != ENOENT && i->second->modTimeResult != EINVAL)
          os << ' ';
      }
      if (i->second->modTimeResult != ENOENT && i->second->modTimeResult != EINVAL) {
        os << i->second->localModTime;
        if (i->second->rebuiltSameTime > i->second->localModTime) {
          os << "\n rebuiltsametime " << i->second->rebuiltSameTime;
          //logMsg << "Actually outputting the rebuiltsametime" << std::endl;
        }
      }
    }
    else
      os << "unknown";
    if (i->second->failTime.tv_sec)
      os << "\n failtime " << i->second->failTime;
    if (i->second->fileflag & FILEFLAG_LOCAL_MODTIME_VALID)
      if (!i->second->modTimeResult)
        os << "\n filesize " << i->second->fileSize;
    if (i->second->buildSeconds)
      os << "\n buildseconds " << i->second->buildSeconds;

    for (std::map<std::string, IncludeVector *>::iterator k = i->second->includeVectorMap.begin();
        k != i->second->includeVectorMap.end(); k++) {
      if (k->second->includeVector.size()) {
        os << "\n folders";
        for (std::vector<std::string>::iterator j = k->second->folders.begin();
            j != k->second->folders.end(); j++) {
          if (j->size())
            os << ' ' << *j;
        }
        os << "\n  includes";
        for (std::vector<BuildFile *>::iterator j = k->second->includeVector.begin();
            j != k->second->includeVector.end(); j++)
          os << ' ' << (*j)->filename;
      }
    }
    os << "\n\n";
  }
  //os << '\n';
}

int BuildFile::isExistenceDep() const {
  //logMsg << logValue(filename) << std::endl;
  //logMsg << logValue(!!ruleset->optionNoSkip) << std::endl;

  if (writeRule && writeRule->commandVector.size()) {
    if (startsWith(writeRule->commandVector[0].c_str(), "mkdir ")
     || startsWith(writeRule->commandVector[0].c_str(), "-mkdir ")) {
      //logMsg << logValue(writeRule->commandVector[0]) << " is an existence dependency" << std::endl;
      //lineabort();
      return 1;
    }
  }

  // 15/9/2025 bumped priority so that "option --fastlink" works
  // in DepTest 3, 4, 5, 6
  if (fileflag & FILEFLAG_EXISTENCE)
    return 1;

  if (ruleset->optionLink)
    return 0;

  for (unsigned i = 0; i < ruleset->existsExt.size(); i++)
    if (endsWith(filename.c_str(), ruleset->existsExt[i].c_str()))
      return 1;

  //logMsg << logValue(fileflag) << std::endl;
  //return fileflag & FILEFLAG_EXISTENCE; // Tested by DepTest15
  return 0;
}

void RuleSet::compareToCache(const RuleSet &cache) {
  // Go through the files. If there are differences in the file times, mark those files
  int foundARuleChange = 0;
  for (std::map<std::string, BuildFile *>::iterator i = fileMap.begin();
      i != fileMap.end(); i++) {
    auto cacheFile = cache.fileMap.find(i->first);
    if (cacheFile == cache.fileMap.end()) {
      //logMsg << "Couldn't find " << logValue(i->first) << std::endl;
    }
    else {
      //logMsg << "Found " << logValue(i->first) << std::endl;
      //logMsg << "Calling calculateModTimes" << std::endl;
      i->second->failTime = cacheFile->second->failTime;
      i->second->localModTime = cacheFile->second->localModTime;
      i->second->rebuiltSameTime = cacheFile->second->rebuiltSameTime;
      i->second->buildSeconds = cacheFile->second->buildSeconds;
      i->second->calculateLocalModTime();
      if ((i->second->fileflag & FILEFLAG_ALIAS)
          != (cacheFile->second->fileflag & FILEFLAG_ALIAS)) {
        // This is an extreme structural change, and anything that uses this file
        // should be thoroughly re-analysed
        // These two cause a recompile when the alias flag goes away, but not when it
        // comes back. Not bad, but the rewritten cache does not have updated includes
        i->second->fileflag |= FILEFLAG_STAT_CHANGED;
        i->second->fileflag |= FILEFLAG_CMD_CHANGED;
        i->second->invalidateState();

        //cacheFile->second->fileflag |= FILEFLAG_STAT_CHANGED;
        //cacheFile->second->fileflag |= FILEFLAG_CMD_CHANGED;
        logMsg << "New alias rule for " << i->second->filename << std::endl;
      }
      if (i->second->fileflag & FILEFLAG_LOCAL_MODTIME_VALID
          && cacheFile->second->fileflag & FILEFLAG_LOCAL_MODTIME_VALID
          && !i->second->modTimeResult
          && !cacheFile->second->modTimeResult) {

        // NEW
        if (cacheFile->second->getFileState("", FOLDERLIST_NONE) == BUILDSTATE_REBUILT_SAME) {
          //i->second->fileState = BUILDSTATE_REBUILT_SAME; // This is unimportant

          //static int setRebuiltSame = getenvint("SETSAME", 1);
          static int setRebuiltSame = 1;
          if (setRebuiltSame) {
            i->second->fileflag |= FILEFLAG_REBUILT_SAME; // This is vital
            //logMsg << "Setting FILEFLAG_REBUILT_SAME " << logValue(i->first) << std::endl;
          }
          //else
            //logMsg << "Not setting FILEFLAG_REBUILT_SAME " << logValue(i->first) << std::endl;
        }

        if (!(i->second->isExistenceDep())) {
          if (senseModTime && i->second->localModTime != cacheFile->second->localModTime) {
            i->second->fileflag |= FILEFLAG_STAT_CHANGED;
            if (getVerbosity(VERBOSITY_STAT, i->second))
              *outputFile->getOStream() << i->first << " modtime changed from "
                << cacheFile->second->localModTime << " to " << i->second->localModTime << '\n';
          }
          if (senseSize && i->second->fileSize != cacheFile->second->fileSize) {
            i->second->fileflag |= FILEFLAG_STAT_CHANGED;
            if (getVerbosity(VERBOSITY_STAT, i->second))
              *outputFile->getOStream() << i->first << " file size changed from "
                << cacheFile->second->fileSize << " to " << i->second->fileSize << '\n';
          }
        }
      }

      if (senseRule) {
        int ruleChanged = 0;
        if (!i->second->writeRule != !cacheFile->second->writeRule) {
          if (!i->second->isExistenceDep()) // ignore rule changes on existence rules
            ruleChanged = 1;
          if (getVerbosity(VERBOSITY_STAT, i->second) || confirmRuleChanges)
            *outputFile->getOStream() << i->first << " build rule removed\n"
                << "was: " << cacheFile->second->writeRule->commandVector << '\n';
        }
        else if (i->second->writeRule) {
          if (i->second->writeRule->commandVector.size()
              != cacheFile->second->writeRule->commandVector.size())
            ruleChanged = 1;
          else
            for (unsigned j = 0; j < i->second->writeRule->commandVector.size(); j++)
              if (i->second->writeRule->commandVector[j]
                  != cacheFile->second->writeRule->commandVector[j]) {
                ruleChanged = 1;
                break;
              }
        }
        if (ruleChanged) {
          if (getVerbosity(VERBOSITY_RULE_CHANGE, i->second) || confirmRuleChanges) {
            if (!i->second->writeRule) {
              //*outputFile->getOStream() << i->first << " build rule removed\n"
                  //<< "was: " << cacheFile->second->writeRule->commandVector << '\n';
              foundARuleChange = 1;
            }
            else {
              const char *bnm = " (but not marked)";
              if (i->second->writeRule->ruleflag & RULEFLAG_MARKED) {
                foundARuleChange = 1;
                bnm = "";
              }
              if (!strcmp(bnm, "") || getVerbosity(VERBOSITY_RULE_CHANGE, i->second))
                if (!cacheFile->second->writeRule)
                  *outputFile->getOStream() << i->first << " build rule added" << bnm << '\n'
                      << "is: " << i->second->writeRule->commandVector << '\n';
                else {
                  /*  15/10/2025
                      This test on commandVector.size() is to work around empty
                      commandVectors in the cache.  But why does that happen?
                      Demonstration:
                      * Remove the test on the commandVector.size()
                      * dep$ dep -vrulechange
                      TODO: Resolve
                   */
                  if (cacheFile->second->writeRule->commandVector.size())
                    *outputFile->getOStream() << i->first << " build rule changed" << bnm << "\n"
                           "from: " << cacheFile->second->writeRule->commandVector << '\n'
                        << "to:   " << i->second->writeRule->commandVector << '\n';
                }
            }
          }
          i->second->fileflag |= FILEFLAG_CMD_CHANGED;
          if (i->second->writeRule)
            i->second->writeRule->invalidateState();
        }
      }

    }
  }
  if (confirmRuleChanges && foundARuleChange) {
    std::cout << "Proceed (Y/n)? " << std::flush;
    std::string reply;
    if (getline(std::cin, reply) && reply.size() && tolower(reply[0]) == 'n')
      exit(0);
  }
}

void RuleSet::combineWithCache(const RuleSet &cache) {
  for (std::map<std::string, BuildFile *>::iterator i = fileMap.begin();
      i != fileMap.end(); i++) {
    auto cacheFile = cache.fileMap.find(i->first);
    if (cacheFile == cache.fileMap.end()) {
      //logMsg << "Couldn't find " << logValue(i->first) << std::endl;
    }
    else {
      //logMsg << "Found " << logValue(i->first) << std::endl;

      // The main question is "Was it executed?" If not, use the cached build rule

      if (i->second->writeRule)
        if (!(i->second->writeRule->buildstate == BUILDSTATE_REBUILT
           || i->second->writeRule->buildstate == BUILDSTATE_REBUILT_SAME)) {
          if (cacheFile->second && cacheFile->second->writeRule)
            i->second->writeRule->commandVector = cacheFile->second->writeRule->commandVector;
        }
    }
  }
}

BuildFile::~BuildFile() {
  delete includesData;
}

IncludeRule::IncludeRule(const char *s):
  syntax(s, std::regex_constants::ECMAScript, STARTSWITH_MATCHES)
{
}

/*   This is the call graph of functions that walk the data structure comprising rules,
 *   source files and include files:
 *
 *                                 RuleSet::processCmd (to produce warnings) REMOVED 30/6/2025
 *                                                 |
 *                           ->getRuleState        |             __
 *                          /    ^   |   v         v            L   \
 *              RuleSet::run     |   |   calculateIncludeModTime     |
 *                          \    |   v   v                      \ _ /
 *                           ->getFileState
 *   RuleSet::outputTo      /
 *   RuleSet::compareToCache
 *   combinedAddIncludes
 *
 *   The functions are recursive, and would be time-consuming to call if their return values
 *   weren't cached.  The caches are:
 *
 *     * getRuleState caches its results
 *     * getFileState caches its results if it is called with FOLDERLIST_SPECIFIC.
 *         If it is called with FOLDERLIST_NONE, it is non-recursive and its value
 *         is not cached.
 *     * calculateIncludeModTime notably does not cache its results, but it is called
 *         by getRuleState which does cache its results, and calculateIncludeModTime
           does not appear in gprof as a hot-spot.
 *
 *   The data structures are built by these functions:
 *
 *     depTreeAddCommand()
 *     L processIncludes()
 *       L combinedAddIncludes()
 *         L treeAddInclude()
 *
 *   The next performance problem to be addressed is revealed by:
 *     $ dep -vtiming
 *     57696 micros reading the caches
 *     189933 micros reading the DepFiles
 *     Targetting all
 *     6824 micros comparing cache and calculating local mod times
 *     8 micros calculating include mod times
 *     dep: all targets are up to date
 *     6089 micros processing the command
 *     6036 micros writing the cache
 *
 *   This can be resolved by skipping the second read if it is not required
 *   It can also be partly resolved resolved by implementing proper import directives
 *   instead of include guards. (DONE, SEE "INCLUDE_ONCE")
 *
 *   The one after that is addressed by loading the cache as an mmap.
 *
 *   Toodo:
 *   * Resolve remaining test failures DONE
 *   * Test on all platforms DONE
 *   * dep/script FOLDERLIST_KLUDGE comparison is irrelevant: remove USING DogFoodTest1b/c INSTEAD
 *   * Remove LogMsgIndenter GONE
 *   * Improve SHOW_TIMINGS and extend it into dep.cpp DONE
 *   - Write a proper unit test <-- DONE
 *   - The remaining performance disparity seems to be due to a loop of calculateIncludeModTime
 *       calls. is this loop really important? NO. REMOVED
 *   - Resolve performance disparity 1: remove loop DONE
 *   - Remove ::inner/::outer stuff DONE
 *   - Remove FOLDERLIST_KLUDGE code DONE
 *   - Abort on all FOLDERLIST_ALL calls DONE
 *   - Use FOLDERLIST_SPECIFIC, not FOLDERLIST_ALL DONE
 *   - Remove all FOLDERLIST_ALL logic DONE
 *   - Remove the FOLDERLIST_parameter entirely from calculateIncludeModTime DONE
 *   - Cut the chaff DONE
 *   - Abort on all FOLDERLIST_NONE calls (not sure this will work) NOPE
 *   - Remove all FOLDERLIST_NONE logic NOPE
 *   - Resolve performance disparity 2: Caching is disabled in calculateIncludeModTime, reinstate NOPE
 */

// Add sources to the BuildFile's include tree and to the rule's source vector

void combinedAddIncludes(BuildFile *bf, Rule *rule, const std::string &filename,
    const std::vector<std::string> *folders, std::fstream &is1, const char *workingDir,
    RuleSet *cacheset) {
  //logMsg << "combinedAddIncludes started on " << filename << std::endl;
  //LogMsgIndenter lmi;
  std::string folderList; // make this from folders above
  if (folders) {
    //logMsg << logValue(*folders) << std::endl;
    for (auto i: *folders) {
      if (folderList.size())
        folderList += " " + i;
      else
        folderList = i;
    }
  }
  else {
    logMsg << "Using an empty folderList" << std::endl;
    //lineabort();
  }
  //logMsg << filename << ' ' << folderList << std::endl;
  RuleSet *ruleset;
  if (bf) {
    if (bf->fileflag & FILEFLAG_TENTATIVE) {
      logMsg << "Returnpoint 1 " << logValue(filename) << std::endl;
      return;
    }
    ruleset = bf->ruleset;
    if (bf->includeVectorMap.count(folderList)) {
      //if ((bf->fileflag & FILEFLAG_FOUND_INCLUDES) && !rule) {
        //logMsg << "Returnpoint 2 " << filename << ' ' << folderList << std::endl;
        // YYY This is it. (Pretty sure.) This should not return.
        // This affects DogFoodTest1a+b, DepTest5, and lots of others
        //return;
      //}
    }
    else {
      //logMsg << logValue(folderList) << " is already there" << std::endl;
      if (bf->includeVectorMap.size())
        bf->fileflag &= ~FILEFLAG_FOUND_INCLUDES;
      //logMsg << "Adding " << logValue(folderList) << " to includeVectorMap of "
      //  << bf->filename << std::endl;
      bf->includeVectorMap[folderList] = new IncludeVector;
    }
    if (folders)
      bf->includeVectorMap[folderList]->folders = *folders;

    // XXX This seems to be premature
    //bf->fileflag |= FILEFLAG_FOUND_INCLUDES;

  }
  else if (rule)
    ruleset = rule->ruleset;
  else {
    logMsg << "Returnpoint 3 " << logValue(filename) << std::endl;
    return;
  }

  std::ostream *os(ruleset->outputFile->getOStream());

  if (!is1.is_open()) {
    if (bf) {
      if (!bf->includesData)
        bf->includesData = new IncludesData;
      bf->includesData->filename = filename;
      if (workingDir)
        bf->includesData->workingDir = workingDir;
      else
        bf->includesData->workingDir = "";
      // XXX This might be better
      bf->fileflag |= FILEFLAG_FOUND_INCLUDES;
    }
    if (ruleset->getVerbosity(VERBOSITY_READ, filename)) {
      *os << "Can't read includes from " << filename << ": file not found\n";
    }
    return;
  }

  const StartPattern *includePattern = 0;
  for (std::vector<IncludeRule>::iterator i = ruleset->includeRuleVector.begin();
      i != ruleset->includeRuleVector.end(); i++) {
    for (std::vector<Pattern>::iterator j = i->filePatternVector.begin();
        j != i->filePatternVector.end(); j++) {
      if (j->matches(filename)) {
        includePattern = &i->syntax;
      }
    }
  }

  if (includePattern) {
    if (ruleset->optionTreeIncludes && cacheset && cacheset->fileMap.count(filename)) {
      // read includes from cache

      BuildFile *cachebf = cacheset->fileMap[filename];

      if (ruleset->outputFile && ruleset->outputFile->getOStream()
          && ruleset->getVerbosity(VERBOSITY_TRACE_STATE, filename)) {
      }
      if (rule && cachebf->getFileState(rule->folderList, FOLDERLIST_SPECIFIC) == BUILDSTATE_OK) {

        // Compare the timestamp to the one in the cache file

        // Using cached stat() calls. This reduces all$ dep from 6.1s to 5.7s

        BuildFile *bf = ruleset->fileMap[filename];
        if (!bf) {
          logMsg << logValue(filename) << std::endl;
          logMsg << logValue(bf) << std::endl;
        }
        bf->calculateLocalModTime();

        if (bf->modTimeResult || bf->localModTime < cacheset->modTime) {
          if (cachebf->includeVectorMap.count(folderList)) {
            for (auto i = cachebf->includeVectorMap[folderList]->includeVector.begin();
                i != cachebf->includeVectorMap[folderList]->includeVector.end(); i++) {
              //logMsg << filename << " includes " << (*i)->filename << std::endl;
              int recurseTree = 0, recurseRule = 0;
              if (bf) {
                logMsg << "combinedAddIncludes calling treeAddInclude("
                  << (*i)->filename << ", " << folderList << ") from cachebf\n";
                recurseTree = bf->treeAddInclude((*i)->filename, folderList);
              }
              if (rule)
                recurseRule = rule->addSource((*i)->filename, 0);
              if (recurseTree || recurseRule) {
                std::fstream is;
                std::string longfilename = ruleset->findFile((*i)->filename,
                  folders, is, filename.c_str(), workingDir);
                logMsg << "Calling combinedAddIncludes\n";
                combinedAddIncludes(recurseTree? bf->includeVectorMap[folderList]->includeVector.back():
                  0, recurseRule? rule: 0, (*i)->filename, folders, is, workingDir, cacheset);
                //logMsg << logValue(recurseTree) << ' ' << logValue(recurseRule)
                // << ' ' << filename << std::endl;
              }
            }
          }

          if (ruleset->getVerbosity(VERBOSITY_READ, filename)) {
            if (rule) {
              *os << "Using cached copy of includes for " << filename << std::endl;
            }
          }

          return;
        }
        else {
          //logMsg << "Had better properly read " << logValue(filename) << std::endl;
        }
      }
    }

    if (ruleset->getVerbosity(VERBOSITY_READ, filename)) {
      //logMsg << "Reading includes from " << filename << std::endl;
      *os << "Reading includes from " << filename << std::endl;
      //if (bf)
        //*os << logValue(bf->filename) << std::endl;
    }

    std::string line;
    int n = 0;
    while (getline(is1, line) && n < ruleset->maxNonIncludes) {
      int pos = -13, len = -17;
      int matches = includePattern->fastMatches(line, &pos, &len);

      //logMsg << logValue(includePattern->s) << ' ' << logValue(line) << std::endl;

      if (matches) {
        if (pos >= 0) {
          std::string includeFilename = line.substr(pos, len);
          std::fstream is;
          std::string longfilename = ruleset->findFile(includeFilename,
            folders, is, filename.c_str(), workingDir);
          int recurseTree = 0, recurseRule = 0;
          if (bf) {
            recurseTree = bf->treeAddInclude(longfilename, folderList);
            // YYY
            //logMsg << "called treeAddInclude(" << longfilename << ", "
              //<< folderList << ") from getline, " << logValue(recurseTree) << std::endl;
          }
          if (rule)
            recurseRule = rule->addSource(longfilename, 0);
          if (recurseTree || recurseRule) {
            //logMsg << "Calling combinedAddIncludes\n";
            combinedAddIncludes(recurseTree? bf->includeVectorMap[folderList]->includeVector.back(): 0,
              recurseRule? rule: 0, longfilename, folders, is, workingDir, cacheset);
          }
          n = 0;
        }
        else {
          n++;
        }
      }
    }
  }
  else {
    if (ruleset->getVerbosity(VERBOSITY_READ, filename)) {
      *os << "#3 Not Reading includes from " << filename << std::endl;
    }
  }
}

void RuleSet::flattenOldFailTimes() {
  std::vector<timespec> failTimeVector;
  for (auto rule = ruleVector.begin(); rule != ruleVector.end(); rule++) {
    for (auto file = (*rule)->targetVector.begin(); file != (*rule)->targetVector.end(); file++) {
      //logMsg << logValue(*file)->filename << ' ' << logValue((*file)->failTime) << std::endl;
      failTimeVector.push_back((*file)->failTime);
    }
  }
  if (failTimeVector.size()) {
    int pos = failTimeVector.size() * 6 / 7;
    std::nth_element(failTimeVector.begin(), failTimeVector.begin() + pos, failTimeVector.end());
    //logMsg << logValue(failTimeVector[pos]) << std::endl;
    timespec min = failTimeVector[pos];
    for (auto rule = ruleVector.begin(); rule != ruleVector.end(); rule++) {
      for (auto file = (*rule)->targetVector.begin(); file != (*rule)->targetVector.end(); file++) {
        if ((*file)->failTime < min) {
          (*file)->failTime.tv_sec = 0;
          (*file)->failTime.tv_nsec = 0;
          //(*file)->failTime = min;
        }
        //logMsg << logValue(*file)->filename << ' ' << logValue((*file)->failTime) << std::endl;
      }
    }
  }
}

Rule::~Rule() {
  delete ruleRunState;
}

int atoidefault(const char *s, int def) {
  if (isdigit(*s))
    return atoi(s);
  return def;
}

void RuleSet::processArgs(AutoCmdLine &cmdline) {
  //lineabort();
  if (cmdline.optionMap.count("-v")) {
    for (auto i = cmdline.optionMap["-v"].begin(); i != cmdline.optionMap["-v"].end(); i++)
      interpretVerbosity(*i);
  }
  if (cmdline.optionMap.count("-q")) {
    interpretVerbosity("-1");
  }
  if (cmdline.optionMap.count("--confirmchanges"))
    confirmRuleChanges = 1;

  optionTreeIncludes = 1; // This should be the fastest but has problems with different -I FIXED
                                  // And fails on full tests on alma due to mingw64 not installed FIXED
  optionRuleIncludes = 0; // This should be the most reliable but has bugs FIXED
                                  // And fails check-acceptance

  if (cmdline.optionMap.count("-r"))
    optionRebuildMarked = 1;
  if (cmdline.optionMap.count("-R"))
    optionRebuildAll = 1;
  if (cmdline.optionMap.count("-d"))
    optionDryrun = 1;
  if (cmdline.optionMap.count("--link")) {
    optionLink = 2;
  }
  if (cmdline.optionMap.count("--fastlink")) {
    if (optionLink == 1)
      optionLink = 0;
  }
  if (cmdline.optionMap.count("--mkdir")) {
    optionMkdir = 2;
  }
  if (cmdline.optionMap.count("--nomkdir")) {
    if (optionMkdir == 1)
      optionMkdir = 0;
  }
  if (cmdline.optionMap.count("--full")) {
    optionMkdir = 2;
    optionLink = 2;
  }
  if (cmdline.optionMap.count("--fast")) {
    if (optionLink == 1)
      optionLink = 0;
    if (optionMkdir == 1)
      optionMkdir = 0;
  }
  if (cmdline.optionMap.count("--fixedorder")) {
    optionFixedOrder = 1;
    maxPipelines = 1; // this helps run things in fixed order for check-acceptance.
  }
  if (cmdline.optionMap.count("--noincludes")) {
    optionTreeIncludes = 0;
    optionRuleIncludes = 0;
  }
  else if (cmdline.optionMap.count("--treeincludes")) {
    optionTreeIncludes = 1;
    optionRuleIncludes = 0;
  }
  else if (cmdline.optionMap.count("--ruleincludes")) {
    optionTreeIncludes = 0;
    optionRuleIncludes = 1;
  }
  else if (cmdline.optionMap.count("--bothincludes")) {
    optionTreeIncludes = 1;
    optionRuleIncludes = 1;
  }
  if (cmdline.optionMap.count("--noreadcache")) {
    readCache = 0;
  }
  if (cmdline.optionMap.count("--readcache")) {
    readCache = 2;
  }
  if (cmdline.optionMap.count("--nowritecache")) {
    writeCache = 0;
  }
  if (cmdline.optionMap.count("--writecache")) {
    writeCache = 2;
  }
  if (cmdline.optionMap.count("--tracefile")) {
    for (unsigned i = 0; i < cmdline.optionMap["--tracefile"].size(); i++)
      tracefileSet.insert(cmdline.optionMap["--tracefile"][i]);
  }
  senseNewer   = 1;
  senseRule    = 1;
  senseModTime = 0;
  senseSize    = 0;
  if (cmdline.optionMap.count("--sense=")) {
    std::string sense = cmdline.optionMap["--sense="].back();
    if (sense.find("all") != std::string::npos) {
      senseModTime = 1;
      senseSize    = 1;
    }
    else {
      senseNewer   = (sense.find("newer")   != std::string::npos);
      senseRule    = (sense.find("rule")    != std::string::npos);
      senseModTime = (sense.find("modtime") != std::string::npos);
      senseSize    = (sense.find("size")    != std::string::npos);
    }
  }
  if (cmdline.optionMap.count("-j"))
    maxPipelines = atoidefault(cmdline.optionMap["-j"].back().c_str(), 1);
  if (cmdline.optionMap.count("--maxnonincludes"))
    maxNonIncludes = atoidefault(cmdline.optionMap["--maxnonincludes"].back().c_str(), 1000);

}

static Cmdspec *depCmdspec = 0;

Cmdspec *getDepCmdspec() {
  if (!depCmdspec) {
    depCmdspec = new Cmdspec(
      "-f--depfile: -h-help--help -V--version -r--rebuild -R--rebuildall\n"
      "-d--dryrun -q--quiet -j--jobs: --treeincludes --ruleincludes\n"
      "--noincludes --bothincludes --maxnonincludes: --sense=newer,rule,modtime,size,all\n"
      "--readcache --noreadcache --nowritecache --writecache --confirmchanges\n"
      "--full --fast --mkdir --nomkdir --link--relink --fastlink\n"
      "--tracefile: --fixedorder -v?",
        0, Position(__FILE__, __LINE__));
    depCmdspec->spec =
      "-f--depfile: -h-help--help -V--version -r--rebuild -R--rebuildall\n"
      "-d--dryrun -q--quiet -j--jobs: --treeincludes --ruleincludes\n"
      "--noincludes --bothincludes --maxnonincludes: --sense=newer,rule,modtime,size,all\n"
      "--readcache --noreadcache --nowritecache --writecache --confirmchanges\n"
      "--full --fast --mkdir --nomkdir --link--relink --fastlink\n"
      "--tracefile: --fixedorder -v0-99\n"
      "-vnormal -vstat -vread -vuptodate -vunselected -vmarked\n"
      "-vstatechange -vstart -vwaitforresult -vtracerun -vfindgenerated -vtracestate\n"
      "-vshoworder -vshowrules -vcheckgraph";
  }
  return depCmdspec;
}

