
// clockwork.h
// Copyright 2022 Matthew Rickard
// This file is part of dep

// dep is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <https://www.gnu.org/licenses/>.

#ifndef CLOCKWORK_H
#define CLOCKWORK_H

#include <string>
#include "Decimal.h"
#include "Integer.h"
#include <vector>
#include "logMsg.h"
#include <strings.h>
#include "stringify.h"
#include "matches.h"
#include <string.h>
#include <map>

#define POSTGRES_EPOCH 730548     // 2000-01-01
#define UNIX_EPOCH     719591     // 1970-01-01
#define SQLITE_EPOCH   UNIX_EPOCH // (This is the advised epoch for integers in SQLite)

#ifndef YEAR_0_SUPPORT
#define YEAR_0_SUPPORT 0
#endif

#if YEAR_0_SUPPORT
#define datediv(a, b) safediv(a, b)
#else
#define datediv(a, b) ((a) / (b))
#endif

void scanNumber(const char *str, short &s);
void scanNumber(const char *str, int &s);
void scanNumber(const char *str, Integer &s);
void scanNumber(const char *str, Decimal &s);
void scanNumber(const char *str, std::string &s);

std::string formatNumber(std::string number, int len);

void getDateModuli(int encoding, int *monMod, int *dayMod);
void getTimeModuli(int encoding, int *hourMod, int *minMod, int *secMod);

template<class Y, class M, class D, class R>
void encodeDate(Y year, M mon, const D &day, int encoding, R *result) {
  if (encoding) {
    int monMod, dayMod;
    getDateModuli(encoding, &monMod, &dayMod);
    *result = (year * monMod + mon - ((monMod == 12)? 1: 0)) * dayMod
      + day - (dayMod == 31? 1: 0);
  }
  else {
    if (mon < 3) {
      year -= 1;
      mon += 13;
    }
    else
      mon++;
    *result = datediv(1461 * year, 4) - datediv(year, 100) + datediv(year, 400)
      + 153 * mon / 5 + day;
  }
}

template<class AD, class Y, class MO, class D>
void decodeDate(AD date, int encoding, Y *year, MO *mon, D *day) {
  if (encoding) {
    int monMod, dayMod;
    getDateModuli(encoding, &monMod, &dayMod);
    if (dayMod == 31) {
      *day = date % dayMod + 1; date /= dayMod;
    }
    else {
      *day = date % dayMod; date /= dayMod;
    }
    if (monMod == 12) {
      *mon = date % monMod + 1; date /= monMod;
    }
    else {
      *mon = date % monMod; date /= monMod;
    }
    *year = date;
  }
  else {
    // Based on the algorithm at
    // https://quasar.as.utexas.edu/BillInfo/JulianDatesG.html
    date += datediv(datediv(4 * date + 145609, 146097) * 3, 4); // stretch to 146100 days
    *year = datediv(4 * date - 489, 1461); // approximate
    *day = date - datediv(1461 * *year, 4); // day-of-year
    *mon = (*day * 5 - 1) / 153; // approximate
    *day -= 153 * *mon / 5; // exact
    *mon = (*mon + 10) % 12 + 1; // exact
    *year += *mon < 3; // exact
  }
}

template<class DT, class E, class Y, class MO, class D, class H, class MI, class S>
void decodeDatetime(DT datetime, const E &epoch, int encoding,
    Y *year, MO *mon, D *day, H *hour, MI *min, S *sec) {
  int hourMod, minMod, secMod;
  getTimeModuli(encoding % 1000, &hourMod, &minMod, &secMod);

  *sec = datetime % secMod; datetime /= secMod;
  *min = datetime % minMod; datetime /= minMod;
  *hour = datetime % hourMod; datetime /= hourMod;
  decodeDate(Integer(datetime + epoch), encoding / 1000, year, mon, day);
}

template<class Y, class MO, class D, class H, class MI, class S>
std::string formatDatetime(const char *format,
    Y year, MO mon, D day, H hour, MI min, S sec, String tzname, String tz) {

  std::string result;
  const char *formatp = format;
  int mIsMonths = 1;
  for (;;) {
    //logMsg << logValue(formatp) << std::endl;
    if (!*formatp)
      break;
    switch (tolower(*formatp)) {
    case 'y':
      result += stringify(year);
      mIsMonths = 1;
      do formatp++; while (tolower(*formatp) == 'y');
      break;
    case 'm': {
        const char *start = formatp;
        do formatp++; while (tolower(*formatp) == 'm');
        int len = formatp - start;
        if (mIsMonths)
          result += formatNumber(stringify(mon), len);
        else
          result += formatNumber(stringify(min), len);
      }
      break;
    case 'd': {
        const char *start = formatp;
        do formatp++; while (tolower(*formatp) == 'd');
        int len = formatp - start;
        result += formatNumber(stringify(day), len);
        mIsMonths = 1;
      }
      break;
    case 'h': {
        const char *start = formatp;
        do formatp++; while (tolower(*formatp) == 'h');
        int len = formatp - start;
        result += formatNumber(stringify(hour), len);
        mIsMonths = 0;
      }
      break;
    case 's': {
        const char *start = formatp;
        do formatp++; while (tolower(*formatp) == 's');
        int len = formatp - start;
        //logMsg << logValue(len) << std::endl;
        std::string tmp = stringify(sec);
        tmp = formatNumber(tmp, len);
        result += tmp;
        mIsMonths = 0;
      }
      break;
    case 'n': {
        int dow;
        encodeDate(year, mon, day, 0, &dow); // hardcoded encoding ok, used for dow calc
        dow %= 7;
        // TOODO: Use a more accessible array
        static const char *dayOfWeekName[] =
          { "Sat", "Sun", "Mon", "Tue", "Wed", "Thu", "Fri" };
        result += dayOfWeekName[dow];
        do formatp++; while (tolower(*formatp) == 'n');
      }
      break;
    case 'z':
      result += tzname.s;
      if (tz.s && *tz.s || !*tzname.s) {
        std::string strtz = tz.s;
        if (*tz.s != '+' && *tz.s != '-')
          result += "+" + strtz;
        else
          result += strtz;
      }
      do formatp++; while (tolower(*formatp) == 'z');
      break;
    case '/':
      result += '/';
      formatp++;
      break;
    case '-':
      result += '-';
      formatp++;
      break;
    case ' ':
      result += ' ';
      formatp++;
      break;
    case 't':
      result += *formatp;
      formatp++;
      break;
    case ':':
      result += ':';
      formatp++;
      mIsMonths = 0;
      break;
    default:
      logMsg << "Can't understand " << logValue(format) << ' ' << logValue(formatp) << std::endl;
      lineabort();
    }
  }
  return result;
}

template<class DT, class H, class MI, class S>
void decodeTime(DT time, int encoding, H *hour, MI *min, S *sec) {
  int hourMod, minMod, secMod;
  getTimeModuli(encoding, &hourMod, &minMod, &secMod);
  *sec = time % secMod; time /= secMod;
  *min = time % minMod; time /= minMod;
  *hour = time % hourMod;
}

template<class DT, class E> std::string formatDatetime(
    const char *format, const DT &datetime, const E &epoch, String offset, int encoding) {
  Integer year;
  short mon, day, hour, min;
  DT sec;
  decodeDatetime(datetime, epoch, encoding, &year, &mon, &day, &hour, &min, &sec);
  return formatDatetime(format, year, mon, day, hour, min, sec, "", offset);
}

template<class DT, class E> std::string formatDate(const char *format,
    const DT &date, E epoch, int encoding) { // headerfile
  Integer year;
  short mon, day;
  decodeDate(date + epoch, encoding, &year, &mon, &day);
  return formatDatetime(format, year, mon, day, 0, 0, 0, "", 0);
}

template<class DT> std::string formatTime(const char *format, const DT &time, int encoding) {
  int hour, min;
  Decimal sec;
  decodeTime(time, encoding, &hour, &min, &sec);
  //logMsg << logValue(format) << ' ' << logValue(time) << ' ' << logValue(encoding)
    //<< ' ' << logValue(sec) << std::endl;
  return formatDatetime(format, 0, 0, 0, hour, min, sec, "", 0);
}

template<class T, class U> void renormalizeValue(T *a, U *b, int from, int to, int tot) {
  if (*a < from || *a > to) {
    int adj = safediv(*a - from, tot);
    (*b) += (adj);
    (*a) -= (adj * tot);
  }
}

extern int daysInMonthArray[];

int daysInMonth(const Integer &y, int m);

// Renormalize a datetime.
// This can easily cope with a single small denormalization.
// It can't easily cope with many large denormalizations spanning multiple defects
// It will also leave very small denormalizations that would require a leap-second check.
// It is intended to be run before a leap-second check

template<class Y, class MO, class D, class H, class MI, class S>
void renormalizeDatetime(Y *year, MO *mon, D *day, H *hour, MI *min, S *sec, int maxSec) {
  if (*sec < 0)
    renormalizeValue(sec, min, 0, 59, 60);
  else
    renormalizeValue(sec, min, 0, maxSec, maxSec + 1);
  renormalizeValue(min, hour, 0, 59, 60);
  renormalizeValue(hour, day, 0, 23, 24);
  renormalizeValue(mon, year, 1, 12, 12);
  if (*day < 1) {
    (*mon)--;
    renormalizeValue(mon, year, 1, 12, 12);
    *day += daysInMonth(*year,*mon);
  }
  else if (*day > daysInMonthArray[*mon]) {
    int dim = daysInMonth(*year, *mon);
    if (*day > dim) {
      (*mon)++;
      renormalizeValue(mon, year, 1, 12, 12);
      *day -= dim;
    }
  }
}

int getLeapSeconds(int yearmon, int *lsyearmon);

template<class A, class B, class C, class D, class E, class F>
int compare(A a1, B b1, C c1, D d1, E e1, F f1, A a2, B b2, C c2, D d2, E e2, F f2) {
  if (a1 < a2) return -1; if (a1 > a2) return 1;
  if (b1 < b2) return -1; if (b1 > b2) return 1;
  if (c1 < c2) return -1; if (c1 > c2) return 1;
  if (d1 < d2) return -1; if (d1 > d2) return 1;
  if (e1 < e2) return -1; if (e1 > e2) return 1;
  if (f1 < f2) return -1; if (f1 > f2) return 1;
  return 0;
}

// Easy algorithm:
// 1 Guess the UTC
// 2 Convert the UTC back to TAI
// 3 If it is not the same as the original, change the guess and go to step 2

template<class Y, class MO, class D, class H, class MI, class S>
void taiToUtc(Y *year, MO *mon, D *day, H *hour, MI *min, S *sec) {
  int lsyearmon;
  *sec -= getLeapSeconds(12 * *year + *mon, &lsyearmon);
  renormalizeDatetime(year, mon, day, hour, min, sec, 59);
  if (lsyearmon == 12 * *year + *mon)
    (*sec)++;
}

template<class Y, class MO, class D, class H, class MI, class S>
void utcToTai(Y *year, MO *mon, D *day, H *hour, MI *min, S *sec) {
  int lsyearmon;
  *sec += getLeapSeconds(12 * *year + *mon, &lsyearmon);
  renormalizeDatetime(year, mon, day, hour, min, sec, 59);
}

// Change offset and abbreviation

// The data type for an offset is a string, or an hours/minutes/seconds triplet.

template<class H, class MI, class S> void scanOffset(String tz, H *hour, MI *min, S *sec) {
  *hour = 0;
  *min = 0;
  *sec = 0;
  if (!*tz.s)
    return;
  int negative = 0;
  if (*tz.s == '+')
    tz.s++;
  else if (*tz.s == '-') {
    tz.s++;
    negative = 1;
  }
  int len = strlen(tz.s);
  if (len >= 2) {
    *hour = 10 * (tz.s[0] - '0') + tz.s[1] - '0';
    if (len >= 4) {
      *min = 10 * (tz.s[2] - '0') + tz.s[3] - '0';
      if (len >= 6)
        scanNumber(tz.s + 4, *sec);
    }
  }
  if (negative) {
    *hour = -*hour;
    *min = -*min;
    *sec = -*sec;
  }
}

template<class Y, class MO, class D, class H, class MI, class S>
void changeOffset(Y *year, MO *mon, D *day, H *hour, MI *min, S *sec,
    std::string *tzname, std::string *tz, String newtzname, String newtz) {
  short oldhours, oldmins, newhours, newmins;
  Decimal oldsecs, newsecs;
  scanOffset(*tz, &oldhours, &oldmins, &oldsecs);
  scanOffset(newtz.s, &newhours, &newmins, &newsecs);

  *hour += newhours - oldhours;
  *min += newmins - oldmins;
  *sec += newsecs - oldsecs;
  //logMsg << *tzname << '-' << newtzname.s << std::endl;
  *tz = newtz;

  if (!strcmp(tzname->c_str(), "TAI")) {
    if (!strcmp(newtzname.s, "TAI"))
      renormalizeDatetime(year, mon, day, hour, min, sec, 59);
    else if (!strcmp(newtzname.s, "GPS")) {
      (*sec) += 19;
      renormalizeDatetime(year, mon, day, hour, min, sec, 59);
    }
    else if (!strcmp(newtzname.s, "TT")) {
      (*sec) += "32.184";
      renormalizeDatetime(year, mon, day, hour, min, sec, 59);
    }
    else
      taiToUtc(year, mon, day, hour, min, sec);
  }
  else if (!strcmp(tzname->c_str(), "GPS")) {
    if (!strcmp(newtzname.s, "TAI")) {
      (*sec) -= 19;
      renormalizeDatetime(year, mon, day, hour, min, sec, 59);
    }
    else if (!strcmp(newtzname.s, "GPS"))
      renormalizeDatetime(year, mon, day, hour, min, sec, 59);
    else if (!strcmp(newtzname.s, "TT")) {
      (*sec) += "51.184";
      renormalizeDatetime(year, mon, day, hour, min, sec, 59);
    }
    else {
      (*sec) -= 19;
      taiToUtc(year, mon, day, hour, min, sec);
    }
  }
  else if (!strcmp(tzname->c_str(), "TT")) {
    if (!strcmp(newtzname.s, "TAI")) {
      (*sec) -= "32.184";
      renormalizeDatetime(year, mon, day, hour, min, sec, 59);
    }
    else if (!strcmp(newtzname.s, "GPS")) {
      (*sec) -= "51.184";
      renormalizeDatetime(year, mon, day, hour, min, sec, 59);
    }
    else if (!strcmp(newtzname.s, "TT")) {
      renormalizeDatetime(year, mon, day, hour, min, sec, 59);
    }
    else {
      (*sec) -= "32.184";
      taiToUtc(year, mon, day, hour, min, sec);
    }
  }
  else {
    // UTC
    if (!strcmp(newtzname.s, "TAI"))
      utcToTai(year, mon, day, hour, min, sec);
    else if (!strcmp(newtzname.s, "GPS")) {
      utcToTai(year, mon, day, hour, min, sec);
      (*sec) += 19;
      renormalizeDatetime(year, mon, day, hour, min, sec, 59);
    }
    else if (!strcmp(newtzname.s, "TT")) {
      utcToTai(year, mon, day, hour, min, sec);
      (*sec) += "32.184";
      renormalizeDatetime(year, mon, day, hour, min, sec, 59);
    }
    else
      if (newsecs)
        // Simple new heuristic: leap seconds are disallowed when timezones are to the second
        renormalizeDatetime(year, mon, day, hour, min, sec, 59);
      else
        renormalizeDatetime(year, mon, day, hour, min, sec, 60);
  }
  *tzname = newtzname.s;
}

// Time zones

struct TimezoneRule {
  short year, mon, day, hour, min, sec;
  std::string tzrule;
};

struct Timezone {
  std::vector<TimezoneRule> v;
};

struct TzMap: std::map<std::string, Timezone *> {
  ~TzMap();
};

//extern std::map<std::string, Timezone *> tzmap;
extern TzMap tzmap;

extern int tzMapInitialized;

void initializeTzMap();

#define compareField(x) if (a.x != b.x) return a.x < b.x? 1: 0

bool operator < (const TimezoneRule &a, const TimezoneRule &b);

  // AEST+1000AEDT+1100,M10.1.0/2,M4.1.0/2

struct TzLexer: Lexer {
  TzLexer() {
    addToken("^[A-Za-z]+", "S", 0);
    addToken("^[+-]?[0-9]+", "N", 0);
    addToken("^,M", ",M", 0);
    addToken("^\\.", ".", 0);
    addToken("^/", "/", 0);
  }
};

int inmodorder(int a, int b, int c);

template<class Y, class MO, class D, class H, class MI, class S, class TZ>
void lookupTimezone(Y *year, MO *mon, D *day, H *hour, MI *min, S *sec,
    std::string *tzname, TZ *tz, String newtzname) {
  if (!tzMapInitialized)
    initializeTzMap();
  //logMsg << logValue(newtzname) << std::endl;
  std::map<std::string, Timezone *>::iterator timezone = tzmap.find(newtzname.s);
  if (timezone == tzmap.end())
    lineabort();
  TimezoneRule now;
  now.year = *year;
  now.mon = *mon;
  now.day = *day;
  now.hour = *hour;
  now.min = *min;
  now.sec = *sec;
  std::vector<TimezoneRule>::iterator rule
    = std::lower_bound(timezone->second->v.begin(), timezone->second->v.end(), now);
  if (rule != timezone->second->v.begin())
    rule--;
  //logMsg << logValue(rule->year) << std::endl;
  //logMsg << logValue(rule->tzrule) << std::endl;

  static TzLexer lexer;
  std::vector<ScanResult> resultVector;
  std::string keyString;
  lexer.scanAll(rule->tzrule.c_str(), resultVector, &keyString);

  //logMsg << logValue(keyString) << std::endl;
  //for (int i = 0; i < resultVector.size(); i++)
    //logMsg << logValue(i) << ' ' << resultVector[i].token << ' ' << resultVector[i].str
      //<< ' ' << resultVector[i].len << std::endl;

  //                0 1 2 3    5   7   9  11   13  15  17  19
  if (keyString == "S N S N ,M N . N . N / N ,M N . N . N / N") {
    std::string stdAbbrev = std::string(resultVector[0].str, resultVector[0].len);
    //int stdOffset = atoi(resultVector[1].str);
    std::string dstAbbrev = std::string(resultVector[2].str, resultVector[2].len);
    //int dstOffset = atoi(resultVector[3].str);
    int getDstDate(int year, ScanResult *sr);
    int dstStart = getDstDate(*year, &resultVector[5]);
    int dstEnd = getDstDate(*year, &resultVector[13]);
    int now;
    encodeDate(*year, *mon, *day, 0, &now);
    now = now * 24 + *hour;
    //logMsg << logValue(dstStart) << ' ' << logValue(now) << ' ' << logValue(dstEnd) << std::endl;
    //logMsg << logValue(rule->tzrule) << std::endl;
    if (inmodorder(dstStart, now, dstEnd)) {
      //logMsg << "Calling changeOffset" << std::endl;
      changeOffset(year, mon, day, hour, min, sec, tzname, tz,
        std::string(resultVector[2].str, resultVector[2].len),
        std::string(resultVector[3].str, resultVector[3].len));
    }
    else {
      //logMsg << "Calling changeOffset" << std::endl;
      changeOffset(year, mon, day, hour, min, sec, tzname, tz,
        std::string(resultVector[0].str, resultVector[0].len),
        std::string(resultVector[1].str, resultVector[1].len));
    }
    //logMsg << logValue(dstStart) << ' ' << logValue(now) << ' ' << logValue(dstEnd) << std::endl;
  }
  else {
    logMsg << logValue(keyString) << std::endl;
    lineabort();
  }
}

//--------------------------------------------------------------------------------------------

template<class Y, class MO, class D, class H, class MI, class S>
void scanDatetime(const char *format, const char *str,
    Y *year, MO *mon, D *day, H *hour, MI *min, S *sec, std::string *tzname, std::string *tz) {
  // Based on ComponentTime::readString (Epoch.cpp)
  *year  = 0;
  *mon = 0;
  *day  = 0;
  *hour  = 0;
  *min = 0;
  *sec  = 0;
  tzname->clear();
  *tz = "";
  const char *strp = str;
  const char *formatp = format;
  int mIsMonths = 1;
  for (;;) {
    //logMsg << logValue(formatp) << std::endl;
    if (!*formatp)
      break;
    switch (tolower(*formatp)) {
    case 'y':
      scanNumber(strp, *year);
      do formatp++; while (tolower(*formatp) == 'y');
      do strp++; while (*strp >= '0' && *strp <= '9');
      mIsMonths = 1;
      break;
    case 'm':
      if (mIsMonths)
        scanNumber(strp, *mon);
      else
        scanNumber(strp, *min);
      do formatp++; while (tolower(*formatp) == 'm');
      do strp++; while (*strp >= '0' && *strp <= '9');
      break;
    case 'd':
      scanNumber(strp, *day);
      do formatp++; while (tolower(*formatp) == 'd');
      do strp++; while (*strp >= '0' && *strp <= '9');
      mIsMonths = 1;
      break;
    case 'h':
      scanNumber(strp, *hour);
      do formatp++; while (tolower(*formatp) == 'h');
      do strp++; while (*strp >= '0' && *strp <= '9');
      mIsMonths = 0;
      break;
    case 's':
      scanNumber(strp, *sec);
      do formatp++; while (tolower(*formatp) == 's');
      do strp++; while (*strp == '.' || *strp >= '0' && *strp <= '9');
      mIsMonths = 0;
      break;
    case 'n':
      do formatp++; while (tolower(*formatp) == 'n');
      do strp++; while (isalpha(*strp));
      break;
    case 'z':
      {
        while (isalpha(*strp)) {
          tzname->push_back(*strp);
          strp++;
        }
        scanNumber(strp, *tz);
        if (*strp == '+' || *strp == '-')
          strp++;
        int len = 0;
        while (isalnum(strp[len]))
          len++;
        do formatp++; while (tolower(*formatp) == 'z');
        do strp++; while (*strp >= '0' && *strp <= '9');
      }
      break;
    case '-':
    case '/':
    case ' ':
    case 't':
      do strp++; while (*strp == *formatp);
      formatp++;
      break;
    case ':':
      do strp++; while (*strp == *formatp);
      formatp++;
      mIsMonths = 0;
      break;
    default:
      logMsg << "Can't understand " << logValue(format) << ' ' << logValue(formatp) << std::endl;
      lineabort();
    }
  }
}

template<class H, class MI, class S, class DT>
void encodeTime(const H &hour, const MI &min, const S &sec, int encoding, DT *result) {
  int hourMod, minMod, secMod;
  getTimeModuli(encoding, &hourMod, &minMod, &secMod);
  *result = (hour * minMod + min) * secMod + sec;
}

template<class Y, class MO, class D, class H, class MI, class S, class E, class DT>
void encodeDatetime(const Y &year, const MO &mon, const D &day,
    const H &hour, const MI &min, const S &sec, const E &epoch, int encoding, DT *result) {
  int hourMod, minMod, secMod;
  getTimeModuli(encoding % 1000, &hourMod, &minMod, &secMod);

  DT intdate; // Toodo: if DT is Decimal, this could just be Integer
  encodeDate(year, mon, day, encoding / 1000, &intdate);
  *result = (((intdate - epoch) * hourMod + hour) * minMod + min) * secMod + sec;
}

template<class DT>
void encodeDate(const char *format, String date, int encoding, DT *result) {
  Integer year;
  short mon, day, hour, min, sec;
  std::string tzname, tz;
  scanDatetime(format, date, &year, &mon, &day, &hour, &min, &sec, &tzname, &tz);
  encodeDate(year, mon, day, encoding, result);
}

template<class DT>
void encodeTime(const char *format, String time, int encoding, DT *result) {
  Integer year;
  short mon, day, hour, min, sec;
  std::string tzname, tz;
  scanDatetime(format, time, &year, &mon, &day, &hour, &min, &sec, &tzname, &tz);
  encodeTime(hour, min, sec, encoding, result);
}

#endif

