
// clockwork.cpp
// Copyright 2022 Matthew Rickard
// This file is part of dep

// dep is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <https://www.gnu.org/licenses/>.

#include "precomp.h"

#include "clockwork.h"

const char *dayName[] = {
  0,
  "Monday",
  "Tuesday",
  "Wednesday",
  "Thursday",
  "Friday",
  "Saturday",
  "Sunday"
};

int daysInMonthArray[] = { 7, 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 };

int daysInMonth(const Integer &y, int m) {
  if (m == 2)
    return y % 4? 28: y % 100? 29: y % 400? 28: 29;
  return daysInMonthArray[m];
}

void scanNumber(const char *str, short &s) {
  s = atoi(str);
}

void scanNumber(const char *str, int &s) {
  s = atoi(str);
}

void scanNumber(const char *str, Integer &s) {
  s.scanString(str);
}

void scanNumber(const char *str, std::string &s) {
  const char *p = str;
  if (*p == '+' || *p == '-')
    p++;
  while (*p >= '0' && *p <= '9' || *p == '.')
    p++;
  //logMsg << logValue(str) << ' ' << (void *)str << ' ' << logValue((void *)p) << std::endl;
  s.assign(str, p - str);
}

void scanNumber(const char *str, Decimal &s) {
  s = str;
}

std::string formatNumber(std::string number, int len) {
  size_t dot = number.find('.');
  if (dot == std::string::npos)
    dot = number.size();
  if (int(dot) < len)
    return std::string(len - dot, '0') + number;
  return number;
}

static int lsMapInitialized = 0;
static std::map<int, int> lsmap;

void initializeLsMap() {
  lsMapInitialized = 1;
  lsmap[12 * 1972 +  6] = 11;
  lsmap[12 * 1972 + 12] = 12;
  lsmap[12 * 1973 + 12] = 13;
  lsmap[12 * 1974 + 12] = 14;
  lsmap[12 * 1975 + 12] = 15;
  lsmap[12 * 1976 + 12] = 16;
  lsmap[12 * 1977 + 12] = 17;
  lsmap[12 * 1978 + 12] = 18;
  lsmap[12 * 1979 + 12] = 19;
  lsmap[12 * 1981 +  6] = 20;
  lsmap[12 * 1982 +  6] = 21;
  lsmap[12 * 1983 +  6] = 22;
  lsmap[12 * 1985 +  6] = 23;
  lsmap[12 * 1987 + 12] = 24;
  lsmap[12 * 1989 + 12] = 25;
  lsmap[12 * 1990 + 12] = 26;
  lsmap[12 * 1992 +  6] = 27;
  lsmap[12 * 1993 +  6] = 28;
  lsmap[12 * 1994 +  6] = 29;
  lsmap[12 * 1995 + 12] = 30;
  lsmap[12 * 1997 +  6] = 31;
  lsmap[12 * 1998 + 12] = 32;
  lsmap[12 * 2005 + 12] = 33;
  lsmap[12 * 2008 + 12] = 34;
  lsmap[12 * 2012 +  6] = 35;
  lsmap[12 * 2015 +  6] = 36;
  lsmap[12 * 2016 + 12] = 37;
}

int getLeapSeconds(int yearmon, int *foundyearmon) {
  if (!lsMapInitialized)
    initializeLsMap();
  std::map<int, int>::iterator i = lsmap.lower_bound(yearmon);
  i--;
  if (i != lsmap.end()) {
    *foundyearmon = i->first;
    return i->second;
  }
  return 10;
}

int tzMapInitialized = 0;

TzMap tzmap;

TzMap::~TzMap() {
  for (std::map<std::string, Timezone *>::iterator i = begin(); i != end(); i++) {
    delete i->second; // this just fixes a minor, but valgrind-visible, memory leak
  }
}

void initializeTzMap() {
  logMsg << "Hello from initializeTzMap" << std::endl;
  tzMapInitialized = 1;
  Timezone *tz = new Timezone();
  tzmap["Australia/Melbourne"] = tz;
  TimezoneRule tr;
  tr.year = -1;
  tr.mon = 1;
  tr.day = 1;
  tr.hour = 0;
  tr.min = 0;
  tz->v.push_back(tr);
  tr.year = 1754;
  tz->v.push_back(tr);
  tr.year = 1854;
  tz->v.push_back(tr);
  tr.year = 1954;
  tz->v.push_back(tr);
  tr.year = 2019;
  // Mm.w.d: m=Month,1=Jan,12=Dec, w=Week,1-5 d=Day,0=Sun,6=Sat
  tr.tzrule = "AEST+10AEDT+11,M10.1.0/2,M4.1.0/2";
  tz->v.push_back(tr);
  tr.year = 2023;
  tz->v.push_back(tr);
  tr.year = 2613;
  tz->v.push_back(tr);
  tr.year = 2913;
  tz->v.push_back(tr);
  tr.year = 3313;
  tz->v.push_back(tr);
  tr.year = 3333;
  tz->v.push_back(tr);
  tr.year = 3334;
  tz->v.push_back(tr);
  tr.year = 3335;
  tz->v.push_back(tr);
  tr.year = 3336;
  tz->v.push_back(tr);
}

int getDstDate(int year, ScanResult *sr) {
  int mon = atoi(sr[0].str);
  int week = atoi(sr[2].str);
  int day = atoi(sr[4].str);
  int hour = atoi(sr[6].str);
  //logMsg << logValue(year) << ' ' << logValue(mon) << ' ' << logValue(week)
    //<< ' ' << logValue(day) << ' ' << logValue(hour) << std::endl;
  int startingPoint;


  encodeDate(year, mon, 0, 0, &startingPoint); // ok, used for day of week calculation
  int weekDayOfFirst = (startingPoint + 6) % 7;

  int result = week * 7 + day - weekDayOfFirst;
  if (day > weekDayOfFirst)
    result -= 7;
  if (result > daysInMonth(year, mon))
    result -= 7;
  result = (startingPoint + result) * 24 + hour;
  return result;
}

int inmodorder(int a, int b, int c) {
  if (a < c)
    return inorder(a, b, c);
  return !inorder(c, b, a);
}

bool operator < (const TimezoneRule &a, const TimezoneRule &b) {
  //logMsg << "Comparing " << a.year << " to " << b.year << std::endl;
  compareField(year);
  compareField(mon);
  compareField(day);
  compareField(hour);
  compareField(min);
  return a.sec < b.sec;
}

  /*  Supported encodings have either:
      * good compression properties
      * good binary or decimal expansions
      * some combination

      Time encodings
      -----------------
       60:   24:60:60
       61:   24:60:61
       64:   24:64:64
       32:   32:64:64
      100:  100:100:100
      256:  256:256:256

      Date encodings
      --------------------
        0:  gregorian days
       31:   12-31 (this won't work)
       32:   12-32
       16:   16-32
      100:  100-100
      256:  256-256
  */

void getDateModuli(int encoding, int *monMod, int *dayMod) {
  //logMsg << logValue(encoding) << std::endl;
  switch (encoding) {
  case 31: *monMod = 12; *dayMod = 31; break;
  case 32: *monMod = 12; *dayMod = 32; break;
  case 16: *monMod = 16; *dayMod = 32; break;
  default: *monMod = *dayMod = encoding;
  }
}

void getTimeModuli(int encoding, int *hourMod, int *minMod, int *secMod) {
  //logMsg << logValue(encoding) << std::endl;
  switch (encoding) {
  case 60: *hourMod = 24; *minMod = 60; *secMod = 60; break;
  case 61: *hourMod = 24; *minMod = 60; *secMod = 61; break;
  case 64: *hourMod = 24; *minMod = 64; *secMod = 64; break;
  case 32: *hourMod = 32; *minMod = 64; *secMod = 64; break;
  default: *hourMod = *minMod = *secMod = encoding;
  }
}

