
// Utils.h
// Copyright 2022 Matthew Rickard
// This file is part of dep

// dep is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <https://www.gnu.org/licenses/>.

#ifndef UTILS_H
#define UTILS_H

#include <stdio.h>
#include <string>
#include "cross_platform.h"
#include "Integer.h"

#ifdef _WIN32
#include <windows.h>
#undef min
#undef max
#endif
#include "stringify.h"

#ifdef __GNUC__
namespace std {
  typedef basic_string<wchar_t> wstring;
};
typedef unsigned long long ULONG64;
typedef unsigned long long UINT64;
#endif

#define square(a) ((a) * (a))

int ReadWholeFile(FILE *f, std::string &contents);
int ReadWholeFile(int fd, std::string &contents);
int ReadWholeFile(String filename, std::string &contents);

void ReplaceAll(std::wstring &s, std::wstring a, std::wstring b);
void ReplaceAll(std::string &s, std::string a, std::string b);
std::string replace(std::string input, const std::string &from, const std::string &to);
ULONG64 FileTime(int year, int month, int day, int hour, int minute, double second);
std::string IsoTime(int year, int month, int day, int hour, int minute, double second);
std::string IsoTime(ULONG64 time);
std::string IsoTime();
#ifdef _WIN32
std::string IsoTime(SYSTEMTIME st);
#endif

std::string recap(std::string a);
std::string RECAP(std::string a);
std::string reCap(std::string a);
std::string ReCap(std::string a);

#ifdef _AFX

struct EasyFont: CFont
{
public:
  BOOL Setup(const wchar_t *typeFace, int weight, int height);
};

#endif

const char *BaseName(const char *filename, const char *delimiters = "\\/");
std::string DirName(const char *filename);

template<typename T> std::basic_string<T> GetField(std::basic_string<T> &line, const T *delimiter = (T *)0) {
  if (delimiter && std::basic_string<T>(delimiter).size()) {
    size_t delimiterPos = line.find(delimiter);
    if (delimiterPos == std::basic_string<T>::npos) {
      std::basic_string<T> result = line;
      line.clear();
      return result;
    }
    std::basic_string<T> result = line.substr(0, delimiterPos);
    line.erase(0, delimiterPos + std::basic_string<T>(delimiter).size());
    return result;
  }
  return line;
}

std::string stripWhitespace(std::string a);
std::string stripLeadingWhitespace(std::string a);

#define TIMER_RESET 1

struct Timer {
  long long startns;
  Timer();
  long long getNanoseconds(int flags = 0);
  long getMicroseconds(int flags = 0);
  long getMilliseconds(int flags = 0);
  long getSeconds(int flags = 0);
  void testSeconds(int seconds);
  void testMilliseconds(int millisecs);
  void start();
};

size_t findNth(const std::string &s, char c, int n);
std::string head(const std::string &s, int n);
std::string demangle(const char *name);
int cint(const char *s);

std::string prefix(const std::string &a, const std::string &b);
int safediv(int a, int b);

typedef void(*sighandler_t)(int);
sighandler_t safeSignal(int signum, sighandler_t handler, int flags);

#ifdef __MINGW32__
#define FCHDIR_AVAILABLE 0
#else
#define FCHDIR_AVAILABLE 0 // should be 1
#endif

#if FCHDIR_AVAILABLE
int getInitialDir();
#else
const char *getInitialDir();
#endif

#define ENABLE_CHECKER 0
#if ENABLE_CHECKER

struct Checker {
  Checker();
  ~Checker();
  virtual void check(const char *file, int line, const char *fn) = 0;
};

struct StreamChecker: Checker {
  StreamChecker(std::ios *s);
  std::ios *stream;
  void check(const char *file, int line, const char *fn);
};

#define checkall() checkall_f(__FILE__, __LINE__, __FUNCTION__);

void checkall_f(const char *file, int line, const char *fn);

#endif

const char *skipWhitespace(const char *s);
const char *skipQuotedString(const char *s, std::string *dequoted);
const char *skipAndDequote(const char *s, std::string *dequoted);

// NOTE: Please use parameterized queries whenever possible
// If some query (e.g. drop table) does not support parameterized queries, use these
// It is vital that user input does not go unquoted, but by secure-defensive programming
// this applies to all variables

std::string sqlQuote(const char *s);
std::string sqlQuote(const std::string &s);

std::string chooseone(const std::string &a, const std::string &b);
std::string pluralize(std::string s);

int safestrcmp(const char *s1, const char *s2);
int safestrcasecmp(const char *s1, const char *s2);

bool checkenv(const char *envvar, const char *expectedValue);

const char *getenvstr(const char *envvar, const char *def);

int getenvint(const char *envvar);

int getenvint(const char *envvar, int defaultValue);

int copyFile(String from, String to);
int copyFile(int from, int to);
int copyFile(std::istream &from, std::ostream &to);
int copyFile(std::istream &from, int to);

int renameFile(const char *from, const char *to);

#define currentLine FileLine(__FILE__, __LINE__)

struct FileLine {
  FileLine() { }
  FileLine(const char *file, int line);
  const char *file;
  int line;
  operator std::string();
};

std::ostream &operator <<(std::ostream &o, const FileLine &sl);

int compare4(int a1, int a2, int a3, int a4, int b1, int b2, int b3, int b4);
int compare2(int a1, int a2, int b1, int b2);
int inorder(int a, int b, int c);
int inorder2(int a1, int a2, int b1, int b2, int c1, int c2);
int inorder4(int a1, int a2, int a3, int a4,
             int b1, int b2, int b3, int b4,
             int c1, int c2, int c3, int c4);

// Keep writing a file until a hard error occurs, circumventing issues
// with temporarily full filesystems, signals and flow control.  If an
// error occurs, the number of bytes written is stored in bytesWritten.

extern size_t bytesWritten;

size_t retryWrite(int fd, const void *buf, size_t count);

std::string changeExt(std::string filename, const char *ext);
std::string removeExt(std::string filename);

struct GeometricSleeper {
  int sleepMillis, step;

  GeometricSleeper(int startMs, int stepMs);
  void sleep();
};

std::ostream &outputRadix(std::ostream &os, long x, int radix);
std::ostream &outputbinary(std::ostream &os, long x);
std::string getRadix(long x, int radix);
std::string getBinary(long x);

std::ostream &outputRadix(std::ostream &os, Integer x, int radix);
std::ostream &outputbinary(std::ostream &os, const Integer &x);
std::string getRadix(const Integer &x, int radix);
std::string getBinary(const Integer &x);

long power(long a, long b);
Integer power(const Integer &a, long b);

#endif

