
// Utils.cpp
// Copyright 2022 Matthew Rickard
// This file is part of dep

// dep is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <https://www.gnu.org/licenses/>.

#include "precomp.h"
#include "Utils.h"
#include <string>
#include <sstream>
#include <iomanip>
#include <iostream>
#include <string.h>
#include <stdlib.h>
#include "logMsg.h"
#include <signal.h>
#include <unistd.h>
#include <cxxabi.h>
#include <fcntl.h>
#include "matches.h"
#include "MacroNames.h"
#include <set>
#include <sys/stat.h>
#include "logMsg.h"

int ReadWholeFile(FILE *f, std::string &contents)
{
  char buf[65536];
  while (size_t len = fread(buf, 1, 65536, f))
    contents.append(buf, len);
  return 0;
}

int ReadWholeFile(int fd, std::string &contents)
{
  char buf[65536];
  ssize_t len;
  while ((len = read(fd, buf, 65536)) > 0) {
    //logMsg << logValue(len) << std::endl;
    contents.append(buf, len);
  }
  return 0;
}

int ReadWholeFile(String filename, std::string &contents) {
  int fd = open(filename.s, O_RDONLY);
  if (fd < 0) {
    //logMsg << "Can't open " << filename.s << std::endl;
    return -1;
  }
  int result = ReadWholeFile(fd, contents);
  close(fd);
  return result;
}

void ReplaceAll(std::string &s, std::string a, std::string b)
{
  //std::cerr << "Was " << s;
  std::string::size_type i = s.find(a);
  while (i != std::string::npos)
  {
    s.replace(i, a.size(), b);
    i = s.find(a, i + b.size());
  }
  //std::cerr << " now is " << s << std::endl;
}

void ReplaceAll(std::wstring &s, std::wstring a, std::wstring b)
{
  std::wstring::size_type i = s.find(a);
  while (i != std::wstring::npos)
  {
    s.replace(i, a.size(), b);
    i = s.find(a, i + b.size());
  }
}

std::string replace(std::string input, const std::string &from, const std::string &to) {
  ReplaceAll(input, from, to);
  return input;
}

#ifdef _AFX

BOOL EasyFont::Setup(const wchar_t *typeFace, int weight, int height)
{
  DeleteObject();
  return CreateFont(
    height,                      // height
    0,                           // width, 0 = default
    0,                           // escapement
    0,                           // orientation
    weight,                      // weight
    FALSE,                       // italic
    FALSE,                       // underline
    FALSE,                       // strikeout
    DEFAULT_CHARSET,             // charset
    OUT_DEFAULT_PRECIS,          // precision
    CLIP_DEFAULT_PRECIS,         // clip precision
    DEFAULT_QUALITY,             // quality
    DEFAULT_PITCH | FF_DONTCARE, // pitch and font family
    typeFace                     // face name
  );
}

#endif

std::string IsoTime(int year, int month, int day, int hour, int minute, double second)
{
  std::ostringstream iso8601DateTime;
  iso8601DateTime << std::setw(4) << std::setfill('0') << year
           << '-' << std::setw(2) << std::setfill('0') << month
           << '-' << std::setw(2) << std::setfill('0') << day
           << 'T' << std::setw(2) << std::setfill('0') << hour
           << ':' << std::setw(2) << std::setfill('0') << minute
           << ':' << std::setw(2) << std::setfill('0') << second;
  return iso8601DateTime.str();
}

#ifdef _WIN32
std::string IsoTime(SYSTEMTIME st)
{
  return IsoTime(st.wYear, st.wMonth, st.wDay, st.wHour, st.wMinute, st.wSecond + 0.001 * st.wMilliseconds);
}

std::string IsoTime(ULONG64 time)
{
  SYSTEMTIME st;
  FileTimeToSystemTime((FILETIME *)&time, &st);
  return IsoTime(st);
}

std::string IsoTime()
{
  SYSTEMTIME st;
  GetLocalTime(&st);
  return IsoTime(st);
}

UINT64 FileTime(int year, int month, int day, int hour, int minute, double second)
{
  SYSTEMTIME st;
  st.wYear = year;
  st.wMonth = month;
  st.wDay = day;
  st.wHour = hour;
  st.wMinute = minute;
  st.wSecond = (WORD)second;
  st.wMilliseconds = (WORD)(1000 * (second - st.wSecond));
  UINT64 result;
  SystemTimeToFileTime(&st, (FILETIME *)&result);
  return result;
}
#endif

std::string DayName(int dayOfWeek)
{
  const char *dayname[7] = { "Sunday", "Monday", "Tuesday", "Wednesday", "Thursday", "Friday", "Saturday" };
  if (dayOfWeek > 0 && dayOfWeek < 8)
    return dayname[dayOfWeek - 1];
  return "Unknown";
}

const char *strrcspn(const char *s, const char *cs)
{
  // Find the _last_ occurrence of any of the characters in cs which
  // appear in the string s.  Modelled loosely after strchr, strrchr
  // and strcspn
  for (const char *p = s + strlen(s) - 1; p >= s; p--)
    if (strchr(cs, *p))
      return p;
  return 0;
}

const char *BaseName(const char *filename, const char *delimiters)
{
  // Return the base name of the filename i.e. the filename with any
  // preceding directory names removed
  const char *lastDelimiter = strrcspn(filename, delimiters);
  if (lastDelimiter)
    return lastDelimiter + 1;
  return filename;
}

std::string DirName(const char *filename)
{
  const char *lastDelimiter = strrcspn(filename, "\\/");
  if (lastDelimiter)
    return std::string(filename, lastDelimiter - filename);
  return "";
}

std::string recap(std::string a) {
  for (std::string::iterator i = a.begin(); i != a.end(); i++)
    *i = tolower(*i);
  return a;
}

std::string reCap(std::string a) {
  for (std::string::iterator i = a.begin(); i != a.end(); i++) {
    *i = tolower(*i);
    break;
  }
  return a;
}

std::string ReCap(std::string a) {
  for (std::string::iterator i = a.begin(); i != a.end(); i++) {
    *i = toupper(*i);
    break;
  }
  return a;
}

std::string RECAP(std::string a) {
  for (std::string::iterator i = a.begin(); i != a.end(); i++)
    *i = toupper(*i);
  return a;
}

std::string stripWhitespace(std::string a) {
  while (a.size() && iswspace(a[a.size() - 1]))
    a.erase(a.size() - 1);
  return a;
}

std::string stripLeadingWhitespace(std::string s) {
  unsigned a;
  for (a = 0; a < s.length() && isspace(s[a]); a++);
  return s.substr(a, s.length() - a);
}

long long getNanosecondsNow() {
#ifdef __unix__
  timespec t;
  clock_gettime(CLOCK_MONOTONIC, &t);
  return 1000000000LL * t.tv_sec + t.tv_nsec;
#elif defined _WIN32
  long long t;
  GetSystemTimeAsFileTime((FILETIME *)&t);
  return 100 * t;
#else
  bork bork bork
#endif
}

Timer::Timer() {
  startns = getNanosecondsNow();
}

long long Timer::getNanoseconds(int flags) {
  long long endns = getNanosecondsNow();
  long long ns(endns - startns);
  if (flags & TIMER_RESET)
    startns = endns;
  return ns;
}

long Timer::getMicroseconds(int flags) {
  return getNanoseconds(flags) / 1000;
}

void Timer::start() {
  startns = getNanosecondsNow();
}

long Timer::getMilliseconds(int flags) {
  return getNanoseconds(flags) / 1000000;
}

long Timer::getSeconds(int flags) {
  return getNanoseconds(flags) / 1000000000;
}

void Timer::testMilliseconds(int Millisecs) {
  int microseconds(getMicroseconds());
  std::cerr << std::setprecision(3) << double(microseconds) / 1000000 << "s ";
  if (microseconds > 1000 * Millisecs)
    std::cerr << "EXCEEDS LIMIT " << Millisecs << "ms\r\n";
}

void Timer::testSeconds(int seconds) {
  int microseconds(getMicroseconds());
  std::cerr << std::setprecision(3) << double(microseconds) / 1000000 << "s ";
  if (microseconds > 1000000 * seconds)
    std::cerr << "EXCEEDS LIMIT " << seconds << "s\r\n";
}

size_t findNth(const std::string &s, char c, int n) {
  if (n <= 0)
    return -1;
  size_t pos(s.find(c, 0));
  while (--n && pos != std::string::npos) {
    size_t pos2(s.find(c, pos + 1));
    if (pos2 == std::string::npos)
      return s.size();
    pos = pos2;
  }
  return pos;
}

std::string head(const std::string &s, int n) {
  if (n <= 0)
    return "";
  size_t pos(findNth(s, '\n', n));
  if (pos == std::string::npos)
    return s;
  std::string result(s.substr(0, pos + 1));
  return result;
}

std::string demangle(const char *mangled) {
  std::size_t len = 0;
  int status = 0;
  return __cxxabiv1::__cxa_demangle(mangled, 0, &len, &status);
  /*
  std::string result, result2;
  while (*name) {
    if (*name >= '0' && *name <= '9') {
      int n(0);
      while (*name >= '0' && *name <= '9') {
        n = 10 * n + (*name - '0');
        name++;
      }
      while (n--)
        result.push_back(*name++);
      continue;
    }
    if (*name == 'P') {
      result2.push_back('*');
      name++;
      continue;
    }
    if (*name == 'K') {
      if (isalnum(result.back()))
        result.push_back(' ');
      result.append("const");
      name++;
      continue;
    }
    if (*name == 'c') {
      if (isalnum(result.back()))
        result.push_back(' ');
      result.append("char");
      name++;
      continue;
    }
    if (*name == 'I') {
      result.push_back('_');
      name++;
      continue;
    }
    if (*name == 'E') {
      name++;
      continue;
    }
    if (*name == 'i') {
      result.append("int");
      name++;
      continue;
    }
    if (!strncmp(name, "Ss", 2)) {
      result.append("std::string");
      name += 2;
      continue;
    }
    std::cerr << __LINE__ << ' ' << ' ' << logValue(result) << ' ' << logValue(name) << std::endl;
    lineabort();
  }
  if (result2.size()) {
    result.push_back(' ');
    result.append(result2);
  }
  return result;
  */
}

int cint(const char *s) {
  //logMsg << "cint: " << logValue(s) << std::endl;
  int result;
  if (*s >= '1' && *s <= '9') {
    result = strtol(s, 0, 10);
    //logMsg << "cint: converted " << logValue(s) << " to " << logValue(result) << std::endl;
    return result;
  }
  if (!strcmp(s, "0")) {
    result = 0;
    //logMsg << "cint: converted " << logValue(s) << " to " << logValue(result) << std::endl;
    return result;
  }
  if (s[0] == '0' && s[1] == 'x') {
    result = strtol(s + 2, 0, 16);
    //logMsg << "cint: converted " << logValue(s) << " to " << logValue(result) << std::endl;
    return result;
  }
  logMsg << "cint: can't do " << logValue(s) << std::endl;
  lineabort();
}

std::string prefix(const std::string &a, const std::string &b) {
  size_t lastslash = b.rfind('/');
  std::string result;
  if (lastslash == std::string::npos)
    result = a + b;
  else
    result = b.substr(0, lastslash + 1) + a + b.substr(lastslash + 1);
  return result;
}

//----------------------------------------------------------------------
// I wonder if you should pre-scan a string, and if it has no regex specials, use ordinary
// string operations? DONE
// Furthermore, I wonder if you should pre-scan a string, and do a string comparison up
// to the first regex special, and if that matches, do the full regex. DONE
//----------------------------------------------------------------------

// Old note: "man 2 signal" says you basically shouldn't use signal, because there
// is no portable way of specifying repeated signal handling.
// Hence safeSignal(), which has the simple API of signal() but
// specifically allows nested signalling.

// New note:

// safeSignal is go-between of signal() and sigaction().
// It is an advanced, reliable version of signal() which has a flags parameter
// It is a simplified version of sigaction which does not have the sa_mask parameter
//
// Some common values of flags are:
// 0                         No special action
// SA_RESTART                BSD-style. Make some system calls restart instead of returning EINTR
//                           (It would be a better idea if it was available on more system calls. It
//                           seems to be undocumented if it is even the default on SIG_IGN.)
// SA_NODEFER                Allows nested signalling, which in turn allows handlers to e.g. log the
//                           signal, communicate with other processes, then reinstate a previous
//                           handler and re-raise the signal (generally stopping the process and
//                           allowing the parent process to see how and why the process stopped)
// SA_RESETHAND              Immediately restore the default signal handler (it is usually better
//                           to do this in code)
// SA_RESETHAND | SA_NODEFER Old UNIX and System V-style (you don't want this)
//
// Additional documentation is spread over signal(2), sigaction(2) and signal(7) man pages

sighandler_t safeSignal(int signum, sighandler_t handler, int flags) {
#ifdef __MINGW32__
  // unfortunately MinGW doesn't do sigaction
  return signal(signum, handler);
#else
  struct sigaction act, oldact;
  act.sa_handler = handler;
  //act.sa_flags = SA_NODEFER; // allow nested signalling
  //act.sa_flags = flags | SA_RESTART; // make some system calls keep running instead of going EINTR
  sigemptyset(&act.sa_mask);
  int result = sigaction(signum, &act, &oldact);
  if (result) {
    logMsg << logValue(result) << std::endl;
    lineabort();
  }
  return oldact.sa_handler;
#endif
}

std::string workingDir() {
  static int maxlen = 32;
  std::string result(maxlen, ' ');
  for (;;) {
    const char *result1 = getcwd((char *)result.data(), maxlen + 1);
    if (result1)
      break;
    if (errno == ERANGE) {
      //logMsg << "Resizing from " << maxlen;
      maxlen *= 3;
      maxlen /= 2;
      //*logStream() << " to " << maxlen << std::endl;
      result.resize(maxlen, ' ');
      continue;
    }
    int error = errno;
    logMsg << logValue(ErrnoToMacroName(error)) << std::endl;
    lineabort();
  }
  result.resize(strlen(result.data()));
  //logMsg << logValue(result) << std::endl;
  return result;
}

#if FCHDIR_AVAILABLE

int getInitialDir() {
  static int initialDir = -99;
  if (initialDir == -99)
    initialDir = open(".", O_RDONLY);
  return initialDir;
}

#else

const char *getInitialDir() {
  static std::string initialDir;
  static int initialized = 0;
  if (!initialized) {
    initialDir = workingDir();
    initialized = 1;
  }
  return initialDir.c_str();
}

#endif

#if ENABLE_CHECKER

std::set<Checker *> checkerSet;

Checker::Checker() {
  checkerSet.insert(this);
}

Checker::~Checker() {
  checkerSet.erase(this);
}

StreamChecker::StreamChecker(std::ios *s): stream(s) {
}

void StreamChecker::check(const char *file, int line, const char *fn) {
  *logStream() << logLine_fn(file, line, fn) << logValue(stream->good()) << std::endl;
  if (!stream->good()) {
    *logStream() << logLine_fn(file, line, fn) << logValue(stream->good()) << std::endl;
    *logStream() << logLine_fn(file, line, fn) << logValue(stream->eof()) << std::endl;
    *logStream() << logLine_fn(file, line, fn) << logValue(stream->fail()) << std::endl;
    *logStream() << logLine_fn(file, line, fn) << logValue(stream->bad()) << std::endl;
    lineabort_f(file, line, fn);
  }
}

void checkall_f(const char *file, int line, const char *fn) {
  for (std::set<Checker *>::iterator i = checkerSet.begin(); i != checkerSet.end(); i++)
    (*i)->check(file, line, fn);
}

#endif

const char *skipWhitespace(const char *s) {
  while (*s == ' ' || *s == '\t')
    s++;
  return s;
}

const char *skipQuotedString(const char *s, std::string *dequoted) {
  char delimiter = *s;
  s++;
  while (*s && *s != delimiter) {
    if (*s == '\\' && (s[1] == delimiter || s[1] == '\\')) {
      s++;
      if (dequoted)
        dequoted->push_back(*s);
    }
    else if (dequoted)
      dequoted->push_back(*s);
    s++;
  }
  if (*s == delimiter)
    s++;
  return s;
}

const char *skipAndDequote(const char *s, std::string *dequoted) {
  if (dequoted)
    dequoted->clear();
  while (*s != ' ' && *s != '\t' && *s != 0) {
    if (*s == '"' || *s == '\'') {
      s = skipQuotedString(s, dequoted);
    }
    else {
      if (dequoted)
        dequoted->push_back(*s);
      s++;
    }
  }
  return s;
}

std::string sqlQuote(const char *s) {
  std::string result;
  for (;;) {
    const char *singlequote = strchr(s, '\'');
    if (!singlequote) {
      result.append(s);
      return result;
    }
    result.append(s, singlequote - s + 1);
    result.push_back('\'');
    s = singlequote + 1;
  }
}

std::string sqlQuote(const std::string &s) {
  return sqlQuote(s.c_str());
}

std::string pluralize(std::string s) {
  if (s.size()) {
    char last = s[s.size() - 1];
    if (last == 's')
      s.append("es");
    else
      s.append("s");
  }
  return s;
}

std::string chooseone(const std::string &a, const std::string &b) {
  if (a.size())
    return a;
  return b;
}

int safestrcasecmp(const char *s1, const char *s2) {
  if (s1 == 0)
    if (s2 == 0)
      return 0;
    else
      return -1;
  if (s2 == 0)
    return 1;
  return strcasecmp(s1, s2);
}

int safestrcmp(const char *s1, const char *s2) {
  //if (s1) logMsg << logValue(s1) << std::endl;
  //if (s2) logMsg << logValue(s2) << std::endl;
  if (s1 == 0)
    if (s2 == 0)
      return 0;
    else
      return -1;
  if (s2 == 0)
    return 1;
  //logMsg << logValue(strcmp(s1, s2)) << std::endl;
  return strcmp(s1, s2);
}

bool checkenv(const char *envvar, const char *value) {
    return safestrcmp(getenv(envvar), value) == 0;
}

int getenvint(const char *envvar) {
  const char *s = getenv(envvar);
  if (!s)
    return 0;
  if (!*s)
    return 0;
  //logMsg << logValue(s) << std::endl;
  if (isdigit(*s) || *s == '-')
    return atoi(s);
  return 1;
}

int getenvint(const char *envvar, int defaultValue) {
  const char *s = getenv(envvar);
  if (!s)
    return defaultValue;
  if (isdigit(*s) || *s == '-')
    return atoi(s);
  return defaultValue;
}

const char *getenvstr(const char *envvar, const char *def) {
  const char *tmp = getenv(envvar);
  return tmp? tmp: def;
}

// old return codes:
// 0 - ok
// 1 - something went wrong

// new return code: number of bytes, or -1 if something went wrong

int renameFile(const char *from, const char *to) {

  // This renames files the Unix way. Windows fails if the target exists. This function removes the target if that occurs and retries.

  int result = rename(from, to);
//#if defined _WIN32 && !defined __CYGWIN__
#if defined _WIN32 || defined __CYGWIN__
  if (result && (errno == EEXIST || errno == EACCES)) { // it was EEXIST, recently added EACCES for Cygwin
    //logMsg << "Let's remove " << to << " and try again." << std::endl;
    int removeResult = remove(to);
    if (removeResult) {
      logMsg << "Now we're really struggling." << std::endl;
      return removeResult;
    }
    result = rename(from, to);
  }
#endif
  return result;
}

int copyFile(std::istream &from, std::ostream &to) {
  char buf[1024];
  int bytes(0);
  do {
    if (from.bad()) {
      lineabort();
      return -1;
    }
    if (to.bad()) {
      lineabort();
      return -1;
    }
    from.read(buf, 1024);
    bytes += from.gcount();
    to.write(buf, from.gcount());
  } while (from.good());
  to.flush();
  return bytes;
}

int copyFile(int fromfd, int tofd) {
  struct stat statbuf;
  if (fstat(tofd, &statbuf))
    return 1;
  char buf[1024];
  int bytes(0);
  int n;
  while ((n = read(fromfd, buf, 1024)) != 0) {
    if (n == -1)
      return -1;
    if (retryWrite(tofd, buf, n) != (size_t)n)
      return -1;
    bytes += n;
  }
  return bytes;
}

int copyFile(std::istream &from, int tofd) {
  struct stat statbuf;
  if (fstat(tofd, &statbuf))
    return 1;
  char buf[1024];
  do {
    if (from.bad()) {
      lineabort();
      return 1;
    }
    from.read(buf, 1024);
    if (retryWrite(tofd, buf, from.gcount()) != (size_t)from.gcount())
      return 1;
  } while (from.good());
  return 0;
}

int copyFile(String from, String to) {
  int fromfd = open(from.s, O_RDONLY);
  if (fromfd == -1) {
    return -1;
  }
  int tofd = creat(to, 0666);
  if (tofd == -1) {
    int error = errno;
    close(fromfd);
    errno = error;
    return -1;
  }
  int result = copyFile(fromfd, tofd);
  if (close(fromfd)) {
    return -1;
  }
  if (close(tofd)) {
    return -1;
  }
  return result;
}

std::ostream &operator <<(std::ostream &o, const FileLine &sl) {
  o << sl.file << ' ' << sl.line;
  return o;
}

FileLine::operator std::string() {
  std::ostringstream o;
  o << *this;
  return o.str();
}

FileLine::FileLine(const char *file, int line): file(file), line(line) {
}

int compare4(int a1, int a2, int a3, int a4, int b1, int b2, int b3, int b4) {
  int result = 0;
       if (a1 < b1) result = -1; else if (a1 > b1) result = 1;
  else if (a2 < b2) result = -1; else if (a2 > b2) result = 1;
  else if (a3 < b3) result = -1; else if (a3 > b3) result = 1;
  else if (a4 < b4) result = -1; else if (a4 > b4) result = 1;
  return result;
}

int compare2(int a1, int a2, int b1, int b2) {
  int result = 0;
       if (a1 < b1) result = -1; else if (a1 > b1) result = 1;
  else if (a2 < b2) result = -1; else if (a2 > b2) result = 1;
  return result;
}

int inorder(int a, int b, int c) {
  return a <= b && b <= c;
}

int inorder2(int a1, int a2, int b1, int b2, int c1, int c2) {
  int result = compare2(a1, a2, b1, b2) < 1
      && compare2(b1, b2, c1, c2) < 1;
  return result;
}

int inorder4(int a1, int a2, int a3, int a4,
             int b1, int b2, int b3, int b4,
             int c1, int c2, int c3, int c4) {
  int result = compare4(a1, a2, a3, a4, b1, b2, b3, b4) < 1
      && compare4(b1, b2, b3, b4, c1, c2, c3, c4) < 1;
  return result;
}

size_t bytesWritten;

size_t retryWrite(int fd, const void *buf, size_t count) {
  bytesWritten = 0;
  for (;;) {
    int result = write(fd, buf, count);
    if (result == -1)
      return (size_t)-1;
    bytesWritten += result;
    if ((size_t)result == count)
      return bytesWritten;
    buf = (const char *)buf + result;
    count -= result;
  }
}

std::string removeExt(std::string filename) {
  int dot = filename.rfind('.');
  int slash = filename.rfind('/');
  if (dot > slash)
    filename.erase(dot);
  return filename;
}

std::string changeExt(std::string filename, const char *ext) {
  filename = removeExt(filename);
  if (ext)
    filename.append(ext);
  return filename;
}

int safediv(int a, int b) {
  //logMsg << logValue(a) << ' ' << logValue(b) << std::endl;
  if (!b)
    return 0;
  div_t divresult = div(a, b);
  if (divresult.rem < 0)
    divresult.quot--;
  return divresult.quot;
}

GeometricSleeper::GeometricSleeper(int startMs, int stepMs): sleepMillis(startMs), step(stepMs) { }

void GeometricSleeper::sleep() {
  //logMsg << "sleeping " << logValue(sleepMillis) << std::endl;
  Sleep(sleepMillis);
  sleepMillis += step;
}

std::ostream &outputRadix(std::ostream &os, long x, int radix) {
  //logMsg << "Checkpoint " << logValue(x) << std::endl;
  if (x < 0)
    lineabort();
  if (x) {
    long bitmask = 1;
    //logMsg << "Checkpoint " << logValue(x) << ' ' << logValue(bitmask) << std::endl;
    while (x >= bitmask) {
      //logMsg << "Checkpoint " << logValue(x) << ' ' << logValue(bitmask) << std::endl;
      bitmask *= radix;
      if (bitmask < 0) {
        logMsg << "Whoopsie " << logValue(bitmask) << std::endl;
        lineabort();
        break;
      }
    }
    //logMsg << "Checkpoint " << logValue(x) << std::endl;
    bitmask /= radix;
    while (bitmask) {
      int digit = x / bitmask;
      if (digit < 0) {
        logMsg << "Whoopsie " << logValue(digit) << std::endl;
        //lineabort();
      }
      if (digit < 10)
        os << digit;
      else
        os << char('a' + (digit - 10));
      x -= digit * bitmask;
      if (x < 0)
        lineabort();
      bitmask /= radix;
    }
  }
  else
    os << '0';
  return os;
}

std::ostream &outputRadix(std::ostream &os, Integer x, int radix) {
  //logMsg << "Checkpoint " << logValue(x) << std::endl;
  if (x < 0)
    lineabort();
  if (x) {
    Integer bitmask = 1;
    //logMsg << "Checkpoint " << logValue(x) << ' ' << logValue(bitmask) << std::endl;
    while (x >= bitmask) {
      //logMsg << "Checkpoint " << logValue(x) << ' ' << logValue(bitmask) << std::endl;
      bitmask *= radix;
      if (bitmask < 0) {
        logMsg << "Whoopsie " << logValue(bitmask) << std::endl;
        lineabort();
        break;
      }
    }
    //logMsg << "Checkpoint " << logValue(x) << std::endl;
    bitmask /= radix;
    while (bitmask) {
      int digit = x / bitmask;
      if (digit < 0) {
        logMsg << "Whoopsie " << logValue(digit) << std::endl;
        //lineabort();
      }
      if (digit < 10)
        os << digit;
      else
        os << char('a' + (digit - 10));
      x -= digit * bitmask;
      if (x < 0)
        lineabort();
      bitmask /= radix;
    }
  }
  else
    os << '0';
  return os;
}

std::ostream &outputbinary(std::ostream &os, long x) {
  return outputRadix(os, x, 2);
}

std::string getRadix(long x, int radix) {
  //logMsg << "Checkpoint " << logValue(x) << std::endl;
  std::ostringstream oss;
  outputRadix(oss, x, radix);
  return oss.str();
}

std::string getRadix(const Integer &x, int radix) {
  //logMsg << "Checkpoint " << logValue(x) << std::endl;
  //lineabort();
  std::ostringstream oss;
  outputRadix(oss, x, radix);
  return oss.str();
}

std::string getBinary(long x) {
  //logMsg << "Checkpoint " << logValue(x) << std::endl;
  return getRadix(x, 2);
}

std::string getBinary(const Integer &x) {
  //logMsg << "Checkpoint " << logValue(x) << std::endl;
  return getRadix(x, 2);
}

long power(long a, long b) {
  long result = 1;
  for (long i = 0; i < b; i++)
    result *= a;
  return result;
}

Integer power(const Integer &a, long b) {
  Integer result = 1;
  for (long i = 0; i < b; i++) {
    result *= a;
    //logMsg << logValue(result) << std::endl;
  }
  return result;
}

