
// ParseTree.h
// Copyright 2015 Matthew Rickard
// This file is part of dep

// dep is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <https://www.gnu.org/licenses/>.

#ifndef PARSETREE_H
#define PARSETREE_H

#include <stdio.h>
#include <string>
#include <vector>

// flex. You don't really have to adjust flex. Here are some standard flex adjustments:

#define YY_SKIP_YYWRAP 1

#ifndef YY_BUF_SIZE
#define YY_BUF_SIZE 16384 // Read buffer size (the biggest value in lex.cpp (for IA64!))
#endif

/* bison. Most bison adjustments are about the stack size, the symbol type (xSTYPE), and
   whether or not the symbol type is "trivial" (i.e. can be reallocated). There is also
   the location type (xLTYPE) but it is practically always four integers, trivial, and
   reallocable.

   If both xSTYPE and xLTYPE are trivial, bison will freely resize (and reallocate) the stack.
   If not, the stack will stay at its initial size, and the parser might fail to parse an
   input.

   Bison determines the symbol reallocability depending on the language.  In C++, it guesses
   that it is non-trivial, but lets you decide with the xSTYPE_IS_TRIVIAL macro.  In C, it
   assumes the symbol type is trivial, and won't be swayed. Neither behaviour is especially
   good.

   So you should set xSTYPE and xSTYPE_IS_TRIVIAL.

   Here are some standard bison adjustments: */

#ifndef YYMAXDEPTH
#define YYMAXDEPTH 20000 // allow stack to grow to 20000 (if stack extension is used)
#endif
#ifndef YYSTACK_USE_ALLOCA
#define YYSTACK_USE_ALLOCA 1 // use alloca not malloc for the parser stack (alloca is better)
#endif
#define YYERROR_VERBOSE 1 // changes "syntax error" to "syntax error, unexpected 'foo'"
#define YYLLOC_DEFAULT(a, b, c) // unused macro, crippled

// Simplified version that outputs only the start position and not the end
#define YY_LOCATION_PRINT(file, loc) fprintf(file, "%d.%d", (loc).first_line, (loc).first_column)

#define HIDDEN_MACRO_CALL        1
#define HIDDEN_MACRO_SUBSTITUTE  2
#define HIDDEN_INCLUDE_DIRECTIVE 4
#define HIDDEN_INCLUDED_CODE     8
#define HIDDEN_QUOTE            16

struct BranchNode;

typedef const char *(*symbolToNameFnType)(int symbol);

extern symbolToNameFnType symbolToNameFnPtr;

struct Position {
  const char *filename;
  int line, col;
  Position();
  Position(const char *filename, int line, int col = 0);
};

std::ostream &operator << (std::ostream &os, const Position &pos);

struct TreeNode {
  BranchNode *parent;
  std::string str(int hidden = 0) const;
  virtual void writeRule(std::ostream &os) const = 0;
  virtual void writeStr(std::ostream &os, int hidden) const = 0;
  virtual void writeTree(std::ostream &os) const = 0;
  const char *getSymbolName();
  std::string getRule() const;
  virtual const TreeNode *subNode(unsigned i) const = 0;
  TreeNode();
  virtual const TreeNode *getTip() const = 0;
  virtual const Position *getPosition() const = 0;
  virtual ~TreeNode();

  short parserSymbol;
  short hidden;
};

struct BranchNode: TreeNode { // based on VectorNode in cppparse/Node.h
  BranchNode(short parserSymbol);
  void pushBack(TreeNode *node);
  void writeRule(std::ostream &os) const;
  void writeStr(std::ostream &os, int hidden) const;
  void writeTree(std::ostream &os) const;
  const TreeNode *subNode(unsigned i) const;
  const TreeNode *getTip() const;
  virtual const Position *getPosition() const;
  virtual ~BranchNode();

  std::vector<TreeNode *> v;
};

struct yyltype {
  int first_line;
  int first_column;
  //int last_line;
  //int last_column;
};

struct LeafNode: TreeNode {
  LeafNode(short parserSymbol, const char *s, const char *filename, yyltype pos);
  void writeRule(std::ostream &os) const;
  void writeStr(std::ostream &os, int hidden) const;
  void writeTree(std::ostream &os) const;
  const TreeNode *subNode(unsigned i) const;
  const TreeNode *getTip() const;
  virtual const Position *getPosition() const;
  virtual ~LeafNode();

  std::string s;
  Position pos;
};

#define MAKENODE() yyval = makeNode(yyr1[yyn], yylen, yytname, yystos, yyssp, yyvsp);

// overloads because bison changes the types depending on the number of tokens/states

TreeNode *makeNode(int r1, int len, const char *const *name,
  const signed char *stos, unsigned char *ssp, TreeNode **vsp);

TreeNode *makeNode(int r1, int len, const char *const *name,
  const signed char *stos, signed char *ssp, TreeNode **vsp);

// This just turned up 21/9/2025

TreeNode *makeNode(int r1, int len, const char *const *name,
  const signed char *stos, short *ssp, TreeNode **vsp);

// For old Bison

TreeNode *makeNode(int r1, int len, const char *const *name,
  const unsigned char *stos, short *ssp, TreeNode **vsp);

#ifdef unix
#include <alloca.h>
#endif

struct yy_buffer_state;
typedef struct yy_buffer_state *YY_BUFFER_STATE;

struct ParseTree {
  ParseTree(YY_BUFFER_STATE (*create_buffer)(FILE *, int),
    void (*delete_buffer)(YY_BUFFER_STATE),
    YY_BUFFER_STATE (*scan_string)(const char *),
    void (*switch_to_buffer)(YY_BUFFER_STATE));
  virtual ~ParseTree();
  int parseFile(const char *filename);
  int parseOpenFile(const char *filename, FILE *f);
  int parseString(const char *s);
  void addOutputFile(const std::string &file);
  void deleteOutputFiles();
  std::vector<std::string> localIncludeDirs, systemIncludeDirs;
  virtual void formatError(std::string error, const yyltype *loc, const std::string &val, int state = -1);
  void formatErrorToStream(std::string error, const yyltype *loc, const std::string &val, int state, std::ostream &os);

  TreeNode *trunk;
  std::string filename;
  int lexemes;
protected:
  void updatePosition(yyltype *loc, char *text);
  int error;
  int last_line, last_column;
private:
  virtual int callParser(YY_BUFFER_STATE buf) = 0;
  YY_BUFFER_STATE (*create_buffer)(FILE *file, int size);
  YY_BUFFER_STATE (*scan_string)(const char *str);
  void (*switch_to_buffer)(YY_BUFFER_STATE buf);
  void (*delete_buffer)(YY_BUFFER_STATE buf);
  std::vector<std::string> outputFiles;
};

#endif

