
// Decimal.cpp
// Copyright 2022 Matthew Rickard

// This file is part of dep
// dep is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <https://www.gnu.org/licenses/>.

#include "precomp.h"
#include "Decimal.h"
#include "cross_platform.h"
#include "logMsg.h"
#include <sstream>
#include <string.h>
#include <gmp.h>

Decimal Decimal::operator - () const {
  Decimal result;
  result.exp = exp;
  result.integer = -integer;
  return result;
}

Decimal::operator short() const {
  Integer result = integer;
  for (int i = 0; i > exp; i--)
    result /= 10;
  for (int i = 0; i < exp; i++)
    result *= 10;
  return result;
}

Decimal::operator int() const {
  Integer result = integer;
  for (int i = 0; i > exp; i--)
    result /= 10;
  for (int i = 0; i < exp; i++)
    result *= 10;
  return result;
}

Decimal::operator long() const {
  Integer result = integer;
  for (int i = 0; i > exp; i--)
    result /= 10;
  for (int i = 0; i < exp; i++)
    result *= 10;
  return result;
}

Decimal::operator long long() const {
  Integer result = integer;
  for (int i = 0; i > exp; i--)
    result /= 10;
  for (int i = 0; i < exp; i++)
    result *= 10;
  return result;
}

bool operator < (const Decimal &a, const Decimal &b) {
  // toodo: try not to actually subtract (NOTE: Tricky)
  Decimal result = a - b;
  return mpz_sgn(result.integer.val) < 0;
}

Decimal::operator bool() const {
  return mpz_sgn(integer.val);
}

Decimal::Decimal(char const *s) {
  int negate = 0;
  if (*s == '-') {
    negate = 1;
    s++;
  }
  else if (*s == '+')
    s++;
  exp = 0;
  while (*s >= '0' && *s <= '9') {
    mpz_mul_ui(integer.val, integer.val, 10);
    mpz_add_ui(integer.val, integer.val, *s - '0');
    s++;
  }
  if (*s == '.') {
    s++;
    while (*s >= '0' && *s <= '9') {
      mpz_mul_ui(integer.val, integer.val, 10);
      mpz_add_ui(integer.val, integer.val, *s - '0');
      s++;
      exp--;
    }
  }
  trim();
  if (negate)
    integer = -integer;
}

Decimal::Decimal(long long val): integer(val), exp(0) {
  trim();
}

Decimal::Decimal(long val): integer(val), exp(0) {
  trim();
}

Decimal::Decimal(unsigned long val): integer(val), exp(0) {
  trim();
}

Decimal::Decimal(int val): integer(val), exp(0) {
  trim();
}

Decimal::Decimal(const Integer &val): integer(val), exp(0) {
  trim();
}

Decimal::Decimal(const Decimal &a): integer(a.integer), exp(a.exp) {
  trim();
}

Decimal::Decimal(): exp(0) {
}

bool operator == (const Decimal &a, const Decimal &b) {
  a.trim();
  b.trim();
  return a.exp == b.exp && a.integer == b.integer;
}

Decimal &Decimal::operator *= (const Decimal &b) {
  integer *= b.integer;
  exp += b.exp;
  return *this;
}

/*

DECIMAL DIVISION

123.45 / 10

The integer is 12345, however you should not just immediately do integer division and
divide that by 10: In integers, 12345 / 10 = 1234.

You really want ... multiply the dividend by 10^X, then divide that, then

*/

Decimal &Decimal::operator /= (const Decimal &b) {
  int extraPlaces = 10; // toodo: calculate this better. It is a complete guess
  for (int i = 0; i < extraPlaces; i++)
    //*this *= 10;
    integer *= 10;
  exp -= extraPlaces;
  //logMsg << *this << " / " << b << std::endl;
  //logMsg << integer << " / " << b.integer << std::endl;
  integer /= b.integer;
  exp -= b.exp;
  if (!integer)
    exp = 0;
  //trim();
  //logMsg << logValue(*this) << std::endl;
  return *this;
}

void Decimal::trim() const {
  // Make sure the integer is as small as possible. Note: this is expensive
  // and unnecessary except for toString() and similar.
  Decimal *that = (Decimal *)this;
  if (mpz_sgn(integer.val)) {
    mpz_t q, r;
    mpz_init(q);
    mpz_init(r);
    for (;;) {
      mpz_fdiv_qr_ui(q, r, integer.val, 10);
      if (mpz_sgn(r))
        break;
      mpz_set(that->integer.val, q);
      that->exp++;
    }
    mpz_clear(q);
    mpz_clear(r);
  }
  else
    that->exp = 0;
}

Decimal &Decimal::operator = (const Decimal &b) {
  integer = b.integer;
  exp = b.exp;
  return *this;
}

Decimal &Decimal::operator -= (const Decimal &b) {
  Decimal bb(b);
  // Maybe do this in full as operator +=
  mpz_neg(bb.integer.val, bb.integer.val);
  return operator +=(bb);
}

Decimal &Decimal::operator += (const Decimal &b) {
  if (b.exp > exp) {
    Decimal bb(b);
    do {
      bb.exp--;
      mpz_mul_ui(bb.integer.val, bb.integer.val, 10);
    }
    while (bb.exp > exp);
    integer += bb.integer;
    return *this;
  }
  while (exp > b.exp) {
    exp--;
      mpz_mul_ui(integer.val, integer.val, 10);
  }
  integer += b.integer;
  return *this;
}

std::string Decimal::toString(int digitsBefore /*= 1*/, int digitsAfter /*= 0*/) const {
  trim();
  std::string s = integer.toString();
  int negate = 0;
  if (s.size() && s[0] == '-') {
    negate = 1;
    s.erase(0, 1);
  }
  int decimalPoint = s.size() + exp;
  for (int i = 0; i < exp + digitsAfter; i++)
    s.push_back('0');
  for (int i = decimalPoint; i < digitsBefore; i++) {
    s.insert(0, "0");
    decimalPoint++;
  }
  if (exp < 0 || digitsAfter)
    s.insert(decimalPoint, ".");
  if (!digitsBefore && s[0] == '0')
    s.erase(0, 1);
  if (negate)
    s.insert(0, "-");
  return s;
}

std::ostream &operator << (std::ostream &os, const Decimal &d) {
  return os << d.toString();
}

// code-generator: cppgen --functions Decimal Integer
Decimal operator + (const Decimal &a, const Decimal &b) {
  Decimal result = a;
  result += b;
  return result;
}
Decimal operator + (const Decimal &a, int b) {
  return a + Decimal(b);
}
Decimal operator + (const Decimal &a, const char *b) {
  return a + Decimal(b);
}
Decimal operator + (const Decimal &a, long b) {
  return a + Decimal(b);
}
Decimal operator + (const Decimal &a, long long b) {
  return a + Decimal(b);
}
Decimal operator + (int a, const Decimal &b) {
  return Decimal(a) + b;
}
Decimal operator + (long a, const Decimal &b) {
  return Decimal(a) + b;
}
Decimal operator + (long long a, const Decimal &b) {
  return Decimal(a) + b;
}
Decimal operator + (const Decimal &a, const Integer &b) {
  return a + Decimal(b);
}
Decimal operator + (const Integer &a, const Decimal &b) {
  return Decimal(a) + b;
}
Decimal operator - (const Decimal &a, const Decimal &b) {
  Decimal result = a;
  result -= b;
  return result;
}
Decimal operator - (const Decimal &a, int b) {
  return a - Decimal(b);
}
Decimal operator - (const Decimal &a, const char *b) {
  return a - Decimal(b);
}
Decimal operator - (const Decimal &a, long b) {
  return a - Decimal(b);
}
Decimal operator - (const Decimal &a, long long b) {
  return a - Decimal(b);
}
Decimal operator - (int a, const Decimal &b) {
  return Decimal(a) - b;
}
Decimal operator - (long a, const Decimal &b) {
  return Decimal(a) - b;
}
Decimal operator - (long long a, const Decimal &b) {
  return Decimal(a) - b;
}
Decimal operator - (const Decimal &a, const Integer &b) {
  return a - Decimal(b);
}
Decimal operator - (const Integer &a, const Decimal &b) {
  return Decimal(a) - b;
}
Decimal operator * (const Decimal &a, const Decimal &b) {
  Decimal result = a;
  result *= b;
  return result;
}
Decimal operator * (const Decimal &a, int b) {
  return a * Decimal(b);
}
Decimal operator * (const Decimal &a, const char *b) {
  return a * Decimal(b);
}
Decimal operator * (const Decimal &a, long b) {
  return a * Decimal(b);
}
Decimal operator * (const Decimal &a, unsigned long b) {
  return a * Decimal(b);
}
Decimal operator * (const Decimal &a, long long b) {
  return a * Decimal(b);
}
Decimal operator * (int a, const Decimal &b) {
  return Decimal(a) * b;
}
Decimal operator * (long a, const Decimal &b) {
  return Decimal(a) * b;
}
Decimal operator * (long long a, const Decimal &b) {
  return Decimal(a) * b;
}
Decimal operator * (const Decimal &a, const Integer &b) {
  return a * Decimal(b);
}
Decimal operator * (const Integer &a, const Decimal &b) {
  return Decimal(a) * b;
}
Decimal operator / (const Decimal &a, const Decimal &b) {
  Decimal result = a;
  result /= b;
  return result;
}
Decimal operator / (const Decimal &a, int b) {
  return a / Decimal(b);
}
Decimal operator / (const Decimal &a, const char *b) {
  return a / Decimal(b);
}
Decimal operator / (const Decimal &a, long b) {
  return a / Decimal(b);
}
Decimal operator / (const Decimal &a, unsigned long b) {
  return a / Decimal(b);
}
Decimal operator / (const Decimal &a, long long b) {
  return a / Decimal(b);
}
Decimal operator / (int a, const Decimal &b) {
  return Decimal(a) / b;
}
Decimal operator / (long a, const Decimal &b) {
  return Decimal(a) / b;
}
Decimal operator / (long long a, const Decimal &b) {
  return Decimal(a) / b;
}
Decimal operator / (const Decimal &a, const Integer &b) {
  return a / Decimal(b);
}
Decimal operator / (const Integer &a, const Decimal &b) {
  return Decimal(a) / b;
}
bool operator == (const Decimal &a, int b) {
  return a == Decimal(b);
}
bool operator == (const Decimal &a, const char *b) {
  return a == Decimal(b);
}
bool operator == (const Decimal &a, long b) {
  return a == Decimal(b);
}
bool operator == (const Decimal &a, long long b) {
  return a == Decimal(b);
}
bool operator == (int a, const Decimal &b) {
  return Decimal(a) == b;
}
bool operator == (long a, const Decimal &b) {
  return Decimal(a) == b;
}
bool operator == (long long a, const Decimal &b) {
  return Decimal(a) == b;
}
bool operator == (const Decimal &a, const Integer &b) {
  return a == Decimal(b);
}
bool operator == (const Integer &a, const Decimal &b) {
  return Decimal(a) == b;
}
bool operator < (const Decimal &a, int b) {
  return a < Decimal(b);
}
bool operator < (const Decimal &a, const char *b) {
  return a < Decimal(b);
}
bool operator < (const Decimal &a, long b) {
  return a < Decimal(b);
}
bool operator < (const Decimal &a, long long b) {
  return a < Decimal(b);
}
bool operator < (int a, const Decimal &b) {
  return Decimal(a) < b;
}
bool operator < (long a, const Decimal &b) {
  return Decimal(a) < b;
}
bool operator < (long long a, const Decimal &b) {
  return Decimal(a) < b;
}
bool operator < (const Decimal &a, const Integer &b) {
  return a < Decimal(b);
}
bool operator < (const Integer &a, const Decimal &b) {
  return Decimal(a) < b;
}
bool operator > (const Decimal &a, const Decimal &b) {
  return b < a;
}
bool operator > (const Decimal &a, int b) {
  return a > Decimal(b);
}
bool operator > (const Decimal &a, const char *b) {
  return a > Decimal(b);
}
bool operator > (const Decimal &a, long b) {
  return a > Decimal(b);
}
bool operator > (const Decimal &a, long long b) {
  return a > Decimal(b);
}
bool operator > (int a, const Decimal &b) {
  return Decimal(a) > b;
}
bool operator > (long a, const Decimal &b) {
  return Decimal(a) > b;
}
bool operator > (long long a, const Decimal &b) {
  return Decimal(a) > b;
}
bool operator > (const Decimal &a, const Integer &b) {
  return a > Decimal(b);
}
bool operator > (const Integer &a, const Decimal &b) {
  return Decimal(a) > b;
}
bool operator != (const Decimal &a, const Decimal &b) {
  return !(a == b);
}
bool operator != (const Decimal &a, int b) {
  return a != Decimal(b);
}
bool operator != (const Decimal &a, const char *b) {
  return a != Decimal(b);
}
bool operator != (const Decimal &a, long b) {
  return a != Decimal(b);
}
bool operator != (const Decimal &a, long long b) {
  return a != Decimal(b);
}
bool operator != (int a, const Decimal &b) {
  return Decimal(a) != b;
}
bool operator != (long a, const Decimal &b) {
  return Decimal(a) != b;
}
bool operator != (long long a, const Decimal &b) {
  return Decimal(a) != b;
}
bool operator != (const Decimal &a, const Integer &b) {
  return a != Decimal(b);
}
bool operator != (const Integer &a, const Decimal &b) {
  return Decimal(a) != b;
}
bool operator <= (const Decimal &a, const Decimal &b) {
  return !(b < a);
}
bool operator <= (const Decimal &a, int b) {
  return a <= Decimal(b);
}
bool operator <= (const Decimal &a, const char *b) {
  return a <= Decimal(b);
}
bool operator <= (const Decimal &a, long b) {
  return a <= Decimal(b);
}
bool operator <= (const Decimal &a, long long b) {
  return a <= Decimal(b);
}
bool operator <= (int a, const Decimal &b) {
  return Decimal(a) <= b;
}
bool operator <= (long a, const Decimal &b) {
  return Decimal(a) <= b;
}
bool operator <= (long long a, const Decimal &b) {
  return Decimal(a) <= b;
}
bool operator <= (const Decimal &a, const Integer &b) {
  return a <= Decimal(b);
}
bool operator <= (const Integer &a, const Decimal &b) {
  return Decimal(a) <= b;
}
bool operator >= (const Decimal &a, const Decimal &b) {
  return !(a < b);
}
bool operator >= (const Decimal &a, int b) {
  return a >= Decimal(b);
}
bool operator >= (const Decimal &a, const char *b) {
  return a >= Decimal(b);
}
bool operator >= (const Decimal &a, long b) {
  return a >= Decimal(b);
}
bool operator >= (const Decimal &a, long long b) {
  return a >= Decimal(b);
}
bool operator >= (int a, const Decimal &b) {
  return Decimal(a) >= b;
}
bool operator >= (long a, const Decimal &b) {
  return Decimal(a) >= b;
}
bool operator >= (long long a, const Decimal &b) {
  return Decimal(a) >= b;
}
bool operator >= (const Decimal &a, const Integer &b) {
  return a >= Decimal(b);
}
bool operator >= (const Integer &a, const Decimal &b) {
  return Decimal(a) >= b;
}
Decimal &Decimal::operator ++ () {
  return *this += 1;
}
// end code-generator
Decimal operator % (const Decimal &a, int b) {
  return a - b * Integer(a / b);
}

