
// CmdLine.cpp
// Copyright 2022 Matthew Rickard
// This file is part of dep

// dep is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <https://www.gnu.org/licenses/>.

#include "precomp.h"

#include "CmdLine.h"
#include <iostream>
#include "logMsg.h"
#include <string.h>

/*
        zero args
   X    one arg
   a    one arg
   aX   one arg
   aXb  two args
   aXbX two args
*/

// -n has a mandatory argument. If empty, when packed, it looks like "-n''" and when
// unpacked, it looks like "-n ''"

// -I has an optional argument. If empty, when packed, it looks like "-n" and when
// unpacked, it looks like "-n ''"

#define ARGFLAG_MANDATORY 1
#define ARGFLAG_OPTIONAL  2

Cmdspec::Cmdspec(const char *spec, int flags, const Position &pos) {
  setup(spec, flags, pos.filename, pos.line);
}

// examples from CmdLineTest133, from CmdLineTest140, DepTest3:

#define logMsgx(file, line, fn) *logStream() << logLine_fn(file, line, fn)

#define mismatch(lexer, token, p) mismatch_f(__FILE__, __LINE__, __FUNCTION__, lexer, token, p)

void mismatch_f(const char *file, int line, const char *fn, Lexer *lexer, int token, const char *p) {
  const char *match = "[Unknown]";
  if (lexer && token < int(lexer->exprVector.size()) && token >= 0)
    match = lexer->exprVector[token].pattern.s.c_str();
  logMsgx(file, line, fn) << "No match for " << p << std::endl;
  if (match == 0) {
    lineabort_f(file, line, fn);
  }
}

// -a -b XFile
// /foo /bar /baz -g--goon -derp -n:Bloopiness -a -I?Gloopiness
// -f: -I:IncludeDir -W: -o:OutputFile -c<AddIncludes> -.* <InputFile>
// -t <InputFile> > :OutputFile
// -t -v -o:OutputFile --defines=:OutputFile,LL <InputFile>

void Cmdspec::setup(const char *argspec, int flags, const char *filename, int line) {
  //logMsg << logValue(spec) << ' ' << logValue(filename) << ' ' << logValue(line) << std::endl;
  this->spec = argspec;
  cmdspecflags = flags;
  int idx = 0;
  std::vector<int> idxv;
  static Lexer lexer;
  static int setup = 0;
  static int dashOption;
  static int slashOption;
  static int whitespace;
  static int colon;
  static int question;
  static int word;
  static int outputRedirect;
  static int mix;
  static int ellipsis;
  static int openSquareBracket;
  static int closeSquareBracket;
  static int barCloseSquareBracket;
  static int doubleDash;
  static int equal;
  if (!setup) {
    setup = 1;
// \x5d is the closing square brace. It is more reliable than \\] or []
#define OPTION_NAME "[a-zA-Z0-9[\\x5d.*,]+"
    dashOption = lexer.addToken("^-+" OPTION_NAME, 0, 0);
    slashOption = lexer.addToken("^/+" OPTION_NAME, 0, 0);
    whitespace = lexer.addToken("^[ \t\r\n]+", 0, 0);
    colon = lexer.addToken("^:", 0, 0);
    question = lexer.addToken("^\\?", 0, 0);
    word = lexer.addToken("^<?%?[a-zA-Z0-9,]+>?", 0, 0);
    outputRedirect = lexer.addToken("^>", 0, 0);
    mix = lexer.addToken("^\\(mix\\)", 0, 0);
    ellipsis = lexer.addToken("^\\.\\.\\.", 0, 0);
    openSquareBracket = lexer.addToken("^\\[", 0, 0);
    closeSquareBracket = lexer.addToken("^]", 0, 0);
    barCloseSquareBracket = lexer.addToken("^\\|.*]", 0, 0);
    doubleDash = lexer.addToken("^--", 0, 0);
    equal = lexer.addToken("^=", 0, 0);
  }

  commandName = "";
  ScanResult scanResult;
  const char *p = spec;
  int startsWithFlag = 0;
  if (cmdspecflags & CMDSPEC_FIXED)
    startsWithFlag = STARTSWITH_FIXED;
  while (*p) {
    const char *p2 = lexer.scan(p, &scanResult);
    if (scanResult.token == outputRedirect) {
      optionVector.push_back(Option(std::string(scanResult.str, scanResult.len), startsWithFlag));
      idxv.push_back(idx);
      Option &option = optionVector.back();

      option.argflag = ARGFLAG_MANDATORY;
      if (*p2) {
        p2 = lexer.scan(p2, &scanResult);
        if (scanResult.token == whitespace) {
          p2 = lexer.scan(p2, &scanResult);
        }
        if (scanResult.token == word) {
          optionVector.back().argName = std::string(scanResult.str, scanResult.len);
        }
      }
      //logMsg << "Checkpoint" << std::endl;
    }
    else if (scanResult.token == mix) {
      cmdspecflags |= CMDSPEC_KEEP_GOING;
    }
    else if (scanResult.token == dashOption || scanResult.token == slashOption) {
      optionVector.push_back(Option(std::string(scanResult.str, scanResult.len), startsWithFlag));
      idxv.push_back(idx);
      p = p2;
      while (*p) {
        p2 = lexer.scan(p, &scanResult);
        //logMsg << logValue(p) << ' ' << logValue(scanResult.token) << std::endl;
        if (scanResult.token == dashOption || scanResult.token == slashOption) {
          optionVector.push_back(Option(std::string(scanResult.str, scanResult.len), startsWithFlag));
          idxv.push_back(idx);
        }
        else if (scanResult.token == whitespace) {
          idx = optionVector.size();
          break;
        }
        else if (scanResult.token == colon
            || scanResult.token == question
            || scanResult.token == equal
            || scanResult.token == word && *p == '%') {
          if (scanResult.token == word) {
          }
          else {
            if (scanResult.token == colon)
              optionVector[idx].argflag = ARGFLAG_MANDATORY;
            else if (scanResult.token == equal) {
              optionVector[idx].argflag = ARGFLAG_OPTIONAL;
              optionVector[idx].optionName.push_back('=');
              //p2++;
            }
            else if (scanResult.token == question)
              optionVector[idx].argflag = ARGFLAG_OPTIONAL;
            else
              lineabort();
            p = p2;
            p2 = lexer.scan(p, &scanResult);
          }
          if (scanResult.token == word) {
            optionVector.back().argName = std::string(scanResult.str, scanResult.len);
            p2 = lexer.scan(p2, &scanResult);
            if (scanResult.token == whitespace)
              idx = optionVector.size();
          }
          else if (scanResult.token == whitespace)
            idx = optionVector.size();
          else if (*p) {
            logMsg << logValue(scanResult.token) << std::endl;
            mismatch(&lexer, scanResult.token, p);
            logMsg << logValue(spec) << std::endl;
            logMsg << logValue(filename) << ' ' << logValue(line) << std::endl;
          }
          break;
        }
        else if (*p) {
          mismatch_f(filename, line, "", &lexer, scanResult.token, p);
          lineabort();
          break;
        }
        p = p2;
      }
    }
    else if (scanResult.token == whitespace) {
      idx = optionVector.size();
    }
    else if (scanResult.token == word) {
      if (1 && p == spec && *p != '%') {
        //logMsg << "The command name seems to be "
          //<< std::string(scanResult.str, scanResult.len) << std::endl;
        commandName = std::string(scanResult.str, scanResult.len);
      }
      else {
        paramVector.push_back(Option()); // This is an argument not an option
        Option *option = &paramVector.back();
        option->argflag = 0;
        option->argName.assign(scanResult.str, scanResult.len);
        idxv.push_back(idx);
        idx = optionVector.size();
      }
    }
    else if (scanResult.token == ellipsis
          || scanResult.token == openSquareBracket
          || scanResult.token == closeSquareBracket
          || scanResult.token == barCloseSquareBracket);
    else if (scanResult.token == doubleDash) {
      cmdspecflags |= (CMDSPEC_COUNT_SECTIONS | CMDSPEC_KEEP_GOING);
    }
    else {
      mismatch(&lexer, scanResult.token, p);
      break;
    }
    p = p2;
  }
  for (unsigned i = 0; i < optionVector.size(); i++)
    optionVector[i].primary = &optionVector[idxv[i]];
  for (unsigned i = 0; i < paramVector.size(); i++)
    paramVector[i].primary = &paramVector[i];
}

// mreallocat - reallocating memory concatenator

char *mreallocat(char *a, int lena, const char *b, int lenb) {
  a = (char *)realloc(a, lena + lenb);
  memcpy(a + lena, b, lenb);
  return a;
}

char *mreallocat3(char *a, int lena, const char *b, int lenb, const char *c, int lenc) {
  a = (char *)realloc(a, lena + lenb + lenc);
  memcpy(a + lena, b, lenb);
  memcpy(a + lena + lenb, c, lenc);
  return a;
}

// reallocat - reallocating string concatenator

char *reallocat(char *a, const char *b) {
  return mreallocat(a, strlen(a), b, strlen(b) + 1);
}

void CmdLineRepacker::reset(int initialState) {
  packable = 0;
  CmdLine::reset(initialState);
}

const char *fragmentTypeToStr(int fragmentType) {
  if (fragmentType == FRAGMENTTYPE_COMMAND)
    return "command";
  if (fragmentType == FRAGMENTTYPE_OPTION)
    return "option";
  if (fragmentType == FRAGMENTTYPE_PARAM)
    return "param";
  return "[unknown fragment type]";
}

void CmdLine::reset(int initialState) {
  currentOption = 0;
  state = initialState;
  haltline = 0;
  paramIdx = 0;
  error = 0;
}

void CmdLineRepacker::processArg(const char *arg) {
  if (flags & CMDPACK_PASSTHROUGH)
    outputArg(arg, KEEP_COPY);
  else
    CmdLine::processArg(arg);
}

void CmdLine::processArg(const char *arg) {
  //if (getenv("DEBUG_CMDLINE"))
    //logMsg << "Processing arg " << arg << std::endl;
  if (currentOption) {
    Fragment fragment;
    fragment.option = currentOption;
    fragment.optionValue = currentOrigStr.c_str();
    fragment.argValue = arg;
    fragment.fragmentType = FRAGMENTTYPE_OPTION;
    fragment.prefix = currentPrefix;
    processFragment(&fragment);
    currentOption = 0;
    return;
  }
  int pos = 0;
  int lenarg = strlen(arg);
  while (pos < lenarg) {
    //if (getenv("DEBUG_CMDLINE"))
      //logMsg << logValue(pos) << std::endl;
    int bestidx = -1;
    int len = 0;
    int fixed = 0;
    if (cmdspec) {
      if (((cmdspec->cmdspecflags & CMDSPEC_KEEP_GOING) || !(state == FRAGMENTTYPE_PARAM))
          || *(arg + pos) == '>') {
        for (unsigned i = 0; i != cmdspec->optionVector.size(); i++) {
          std::string option = cmdspec->optionVector[i].optionName;
          if (debug)
            logMsg << logValue(option) << ' ' << logValue(arg) << std::endl;
          int optionOffset = 0, argOffset = 0;
          if (pos) {
            optionOffset = 1;
            //logMsg << "bumping optionOffset" << std::endl;
          }
          if (cmdspec->cmdspecflags & CMDSPEC_ANY_PREFIX && optionOffset != 1) {
            argOffset = 1;
            optionOffset = 1;
            //logMsg << "bumping argOffset and optionOffset" << std::endl;
          }
          int lenx = -999;
          int fixedx = -999;

          const StartPattern *sp = optionOffset? &cmdspec->optionVector[i].pattern1:
            &cmdspec->optionVector[i].pattern0;
          int swResult = sp->patternStartsWith(arg + pos + argOffset,
            option.c_str() + optionOffset, option.size() - optionOffset, &lenx);
          fixedx = !cmdspec->optionVector[i].pattern0.useRegex;
          if (swResult) {
            lenx += optionOffset;
            if (fixedx > fixed || fixedx == fixed && lenx > len) {
              len = lenx;
              fixed = fixedx;
              bestidx = i;
              if (debug)
                logMsg << logValue(bestidx) << std::endl;
            }
          }
        }
      }
    }
    if (debug)
      logMsg << logValue(bestidx) << std::endl;
    Fragment fragment;
    fragment.prefix = *arg;
    if (bestidx == -1) {
      fragment.option = 0;
      fragment.optionValue = "";
      fragment.argValue = arg + pos;
      if (state == FRAGMENTTYPE_COMMAND)
        fragment.fragmentType = FRAGMENTTYPE_COMMAND;
      else {
        fragment.fragmentType = FRAGMENTTYPE_PARAM;
        if (cmdspec && cmdspec->paramVector.size()) {
          fragment.option = &cmdspec->paramVector[paramId];
          if (paramId < int(cmdspec->paramVector.size()) - 1)
            paramId++;
        }
      }
      if (debug)
        logMsg << "Checkpoint" << std::endl;
      processFragment(&fragment);
      if (state == FRAGMENTTYPE_COMMAND)
        state = FRAGMENTTYPE_OPTION;
      else if (state == FRAGMENTTYPE_OPTION)
        state = FRAGMENTTYPE_PARAM;
      else if (state == FRAGMENTTYPE_PARAM);
      else
        lineabort();
      return;
    }
    if (state == FRAGMENTTYPE_COMMAND)
      state = FRAGMENTTYPE_OPTION;
    fragment.option = &cmdspec->optionVector[bestidx];
    std::string optionStr;
    if (pos) {
      optionStr.assign(arg, 1);
      len--;
      optionStr.append(arg + pos, len);
    }
    else
      optionStr.assign(arg + pos, len);
    if (debug)
      logMsg << logValue(optionStr) << std::endl;
    fragment.optionValue = optionStr.c_str(); // XXX this is messy
    fragment.fragmentType = FRAGMENTTYPE_OPTION;
    pos += len;
    if (fragment.option->primary->argflag == ARGFLAG_MANDATORY) {
      if (arg[pos]) {
        fragment.argValue = arg + pos;
        processFragment(&fragment);
        return;
      }
      currentOption = fragment.option;
      currentOrigStr = fragment.optionValue;
      currentPrefix = *arg;
      return;
    }
    if (fragment.option->primary->argflag == ARGFLAG_OPTIONAL && arg[pos]) {
      fragment.argValue = arg + pos;
      if (debug)
        logMsg << logValue(fragment.argValue) << std::endl;
      processFragment(&fragment);
      return;
    }
    if (pos == lenarg || optionStr.size() < 3) {
      fragment.argValue = 0;
      processFragment(&fragment);
    }
    else {
      error = 1;
      //if (getenv("DEBUG_CMDLINE"))
        //logMsg << "Dot doing arg " << logValue(arg) << ' ' << logValue(optionStr) << std::endl;
      // lineabort();
    }
  }
  return;
}

void CmdLine::processArgv(int initialState, int argc, const char *const argv[]) {
  this->argv = argv;
  reset(initialState);
  for (int i = 0; i < argc; i++ && !haltline) {
    if (debug)
      logMsg << logValue(paramIdx) << std::endl;
    processArg(argv[i]);
    if (state != FRAGMENTTYPE_PARAM)
      paramIdx = i + 1;
    if (!strcmp(argv[i], "--") && !sectionBreak)
      sectionBreak = i;
    if (debug)
      logMsg << logValue(i) << ' ' << logValue(state) << std::endl;
  }
}

void CmdLine::processCmdline(int initialState, const char *argcmdline, int divider) {
  if (debug)
    logMsg << "Processing a cmdline " << logValue(argcmdline) << std::endl;
  reset(initialState);
  // one of these days: combine these two algorithms somehow
  position = argcmdline;
  if (divider == DIVIDER_WHITESPACE) {
    while (*position && !haltline) {
      std::string s;
      const char *endpos = skipAndDequote(position, &s);
      position = skipWhitespace(endpos);
      //logMsg << logValue(s) << std::endl;
      processArg(s.c_str());
      if (!*endpos)
        break;
    }
  }
  else {
    const char *x;
    while ((x = strchr(position, divider)) && !haltline) {
      processArg(std::string(position, x - position).c_str());
      position = x + 1;
    }
    if (*position)
      processArg(position);
  }
}

CmdLine::CmdLine(const Cmdspec *cmdspec): sectionBreak(0), cmdspec(cmdspec), paramId(0) {
  debug = 0;
}

void CmdLineArgvMaker::outputArg(const char *arg, int keepnote) {
  if (flags & CMDPACK_COPY) {
    char *arg2 = strdup(arg);
    v.push_back(arg2);
  }
  else if (keepnote == KEEP_COPY) {
    char *arg2 = strdup(arg);
    v.push_back(arg2);
    v2.push_back(arg2);
  }
  else if (keepnote == KEEP_FREE) {
    v.push_back((char *)arg);
    v2.push_back((char *)arg);
  }
  else if (keepnote == KEEP_NONE) {
    //v.push_back((char *)arg);
    lineabort();
  }
  else
    lineabort();
}

CmdLineArgvMaker::~CmdLineArgvMaker() {
  for (std::vector<char *>::iterator i = v2.begin(); i != v2.end(); i++)
    free(*i);
}

void CmdLineStringMaker::outputArg(const char *newarg, int keepnote) {
  if (arg) {
    if (cmdline.size())
      cmdline.push_back(' ');
    int others = 0, singles = 0, doubles = 0;
    for (const char *p = arg; *p; p++)
      if (*p == ' ' || *p == '\\')
        others++;
      else if (*p == '\'')
        singles++;
      else if (*p == '\"')
        doubles++;
    char quoteChar = 0;
    if (others || singles || doubles || !*arg)
      if (doubles > singles)
        quoteChar = '\'';
      else
        quoteChar = '"';
    if (quoteChar)
      cmdline.push_back(quoteChar);
    for (const char *p = arg; *p; p++) {
      if (*p == quoteChar)
        cmdline.push_back('\\');
      if (*p == '\\' && (p[1] == '\\' || !p[1] || p[1] == quoteChar))
        cmdline.push_back('\\');
      cmdline.push_back(*p);
    }
    if (quoteChar)
      cmdline.push_back(quoteChar);
    free((char *)arg);
  }
  if (keepnote == KEEP_COPY)
    arg = strdup(newarg);
  else
    arg = newarg;
}

CmdLineStringMaker::CmdLineStringMaker(int flags, Cmdspec *cmdspec):
  CmdLineRepacker(flags, cmdspec), arg(0) {
}

char *strdupcat(const char *a, int lena, const char *b, int lenb) {
  char *result = (char *)malloc(lena + lenb + 1);
  memcpy(result, a, lena);
  memcpy(result + lena, b, lenb + 1);
  return result;
}

char *CmdLineArgvMaker::getBack() {
  return v.back();
}

void CmdLineArgvMaker::setBack(char *a) {
  for (std::vector<char *>::iterator i = v2.begin(); i != v2.end(); i++)
    if (*i == v.back()) {
      v2.erase(i);
      break;
    }
  v.back() = a;
  v2.push_back(a);
}

char *CmdLineStringMaker::getBack() {
  return (char *)arg;
}

void CmdLineStringMaker::setBack(char *a) {
  arg = a;
}

void CmdLineRepacker::processFragment(const Fragment *fragment) {
  if (flags & CMDPACK_PACK) {
    if (packable && fragment->option && (getBack()[0] == fragment->prefix)) {
      if (fragment->argValue) {
        setBack(mreallocat3(getBack(), strlen(getBack()),
          fragment->optionValue + 1, strlen(fragment->optionValue + 1),
          fragment->argValue, strlen(fragment->argValue) + 1));
      }
      else
        setBack(mreallocat(getBack(), strlen(getBack()),
          fragment->optionValue + 1, strlen(fragment->optionValue)));
    }
    else
      if (fragment->option)
        if (fragment->argValue) {
          outputArg(strdupcat(fragment->optionValue,
            strlen(fragment->optionValue), fragment->argValue,
            strlen(fragment->argValue)), KEEP_FREE);
        }
        else {
          outputArg(fragment->optionValue, KEEP_COPY);
        }
      else
        outputArg(fragment->argValue, KEEP_COPY);
    if (fragment->option && fragment->option->primary->argflag == 0
        && strlen(fragment->optionValue) < 3) {
      //logMsg << logValue(fragment->optionValue) << std::endl;
      packable = 1;
    }
    else
      packable = 0;
  }
  else if (flags & CMDPACK_UNPACK)
    if (fragment->option)
      if (fragment->argValue)
        if (fragment->option->argflag == ARGFLAG_MANDATORY) {
          outputArg(fragment->optionValue, KEEP_COPY);
          outputArg(fragment->argValue, KEEP_COPY);
        }
        else
          outputArg(strdupcat(fragment->optionValue,
            strlen(fragment->optionValue), fragment->argValue,
            strlen(fragment->argValue)), KEEP_FREE);
      else {
        outputArg(fragment->optionValue, KEEP_COPY);
      }
    else
      outputArg(fragment->argValue, KEEP_COPY);
  else
    lineabort();
}

void cmdlineToArgv(const char *cmdline, std::vector<char *> &argv, int divider) {
  //logMsg << logValue(cmdline) << std::endl;
  CmdLineArgvMaker clta(CMDPACK_PASSTHROUGH | CMDPACK_COPY, 0);
  clta.processCmdline(FRAGMENTTYPE_COMMAND, cmdline, divider);
  //for (int i = 0; i < clta.v.size(); i++)
    //logMsg << logValue(i) << ' ' << logValue(clta.v[i]) << std::endl;
  std::swap(argv, clta.v);
}

std::string argvToCmdline(int argc, const char *const argv[]) {
  //for (int i = 0; i < argc; i++)
    //logMsg << logValue(i) << ' ' << logValue(argv[i]) << std::endl;
  CmdLineStringMaker atcl(CMDPACK_PASSTHROUGH, 0);
  atcl.processArgv(FRAGMENTTYPE_COMMAND, argc, argv);
  //logMsg << logValue(atcl.getCmdline()) << std::endl;
  return atcl.getCmdline();
}

CmdLineArgvMaker::CmdLineArgvMaker(int flags, Cmdspec *cmdspec):
  CmdLineRepacker(flags, cmdspec) {
}

CmdLineRepacker::CmdLineRepacker(int flags, Cmdspec *cmdspec):
  CmdLine(cmdspec), flags(flags) {
}

std::string unpackToCmdline(Cmdspec *cmdspec, int argc, const char *const argv[], int flags) {
  CmdLineStringMaker cv(flags | CMDPACK_UNPACK, cmdspec);
  cv.processArgv(FRAGMENTTYPE_COMMAND, argc, argv);
  return cv.getCmdline();

}

std::string packToCmdline(Cmdspec *cmdspec, int argc, const char *const argv[], int flags) {
  CmdLineStringMaker cv(flags | CMDPACK_PACK, cmdspec);
  cv.processArgv(FRAGMENTTYPE_COMMAND, argc, argv);
  return cv.getCmdline();
}

const std::string &CmdLineStringMaker::getCmdline() {
  //logMsg << logValue(!!arg) << std::endl;
  if (arg)
    outputArg(0, KEEP_NONE);
  //logMsg << logValue(cmdline) << std::endl;
  return cmdline;
}

Option::Option(String optionName, int flags):
    optionName(optionName),
    pattern0(optionName.s, std::regex::ECMAScript, flags),
    pattern1(optionName.s + 1, std::regex::ECMAScript, flags),
    primary(0),
    argflag(0) {
  //logMsg << "constructing an option, " << logValue(flags) << std::endl;
}

Option::Option():
  pattern0("", std::regex_constants::ECMAScript, 0),
  pattern1("", std::regex_constants::ECMAScript, 0),
  primary(0), argflag(0) {
}

void Cmdspec::outputTo(std::ostream &os, int outputComments) {
  if (cmdspecflags & CMDSPEC_KEEP_GOING)
    os << " (mix)";
  for (auto i = optionVector.begin(); i != optionVector.end(); i++) {
    if (i->optionName != ">") {
      os << ' ' << i->optionName;
      if (i->primary) {
        if (i->primary && i->primary != &*i)
          os << " (" << i->primary->optionName << ')';
        if (i->primary->argflag == ARGFLAG_OPTIONAL) {
          if (i->primary->optionName.back() != '=')
            os << '?';
        }
        else if (i->primary->argflag == ARGFLAG_MANDATORY)
          os << ':';
      }
      if (i->argName.size())
        os << i->argName;
    }
  }
  for (auto i = paramVector.begin(); i != paramVector.end(); i++) {
    os << i->optionName;
    if (i->primary != &*i)
      os << " (" << i->primary->optionName << ')';
    if (i->primary->argflag == ARGFLAG_OPTIONAL)
      os << " (optional)";
    else if (i->primary->argflag == ARGFLAG_MANDATORY)
      os << " (mandatory)";
    if (i->argName.size())
      os << ' ' << i->argName;
  }
  for (auto i = optionVector.begin(); i != optionVector.end(); i++) {
    // This is pretty crappy. This whole loop is a copy of the first one, except for this:
    if (i->optionName == ">") {
      os << ' ' << i->optionName;
      if (i->primary) {
        if (i->primary && i->primary != &*i)
          os << " (" << i->primary->optionName << ')';
        if (i->primary->argflag == ARGFLAG_OPTIONAL)
          os << " (optional arg)";
        //else if (i->primary->argflag == ARGFLAG_MANDATORY)
          //os << ':';
      }
      if (i->argName.size())
        os << i->argName;
    }
  }
  os << std::endl;
  if (outputComments) {
    for (auto i = optionVector.begin(); i != optionVector.end(); i++) {
      os << "#Option " << i->optionName;
      if (i->primary) {
        if (i->primary && i->primary != &*i)
          os << " (" << i->primary->optionName << ')';
        if (i->primary->argflag == ARGFLAG_OPTIONAL)
          os << " (optional arg)";
        else if (i->primary->argflag == ARGFLAG_MANDATORY)
          os << " (requires arg)";
      }
      if (i->argName.size())
        os << ' ' << i->argName;
      os << std::endl;
    }
    for (auto i = paramVector.begin(); i != paramVector.end(); i++) {
      if (i->optionName.size())
        os << "#Param " << i->optionName;
      else
        os << "#Param";
      if (i->primary != &*i)
        os << " (" << i->primary->optionName << ')';
      if (i->primary->argflag == ARGFLAG_OPTIONAL)
        os << " (optional)";
      else if (i->primary->argflag == ARGFLAG_MANDATORY)
        os << " (mandatory)";
      if (i->argName.size())
        os << ' ' << i->argName;
      os << std::endl;
    }
  }
}

AutoCmdLine::AutoCmdLine(Cmdspec *cmdspec): CmdLine(cmdspec), sections(1) {
}

void AutoCmdLine::processFragment(const Fragment *f) {
  //if (getenv("DEBUG_CMDLINE")) {
    //logMsg << "Processing fragment " << logValue(f->optionValue) << std::endl;
    //if (f->argValue)
      //logMsg << logValue(f->argValue) << std::endl;
  //}
  if (f->fragmentType == FRAGMENTTYPE_OPTION) {
    if (f->argValue) {
      optionMap[f->option->primary->optionName].push_back(f->argValue);
      optionVector.push_back(std::pair(f->option->primary->optionName, f->argValue));
    }
    else {
      optionMap[f->option->primary->optionName].push_back("");
      optionVector.push_back(std::pair(f->option->primary->optionName, ""));
    }
  }
  else if (f->fragmentType == FRAGMENTTYPE_PARAM) {
    if (cmdspec->cmdspecflags & CMDSPEC_COUNT_SECTIONS) {
      if (!strcmp(f->argValue, "--")) {
        sections++;
      }
    }
    else {
      haltline = 1;
      if (*f->argValue == '-' || *f->argValue == '/')
        error = 1;
    }
  }
}

std::ostream &operator <<(std::ostream &os, const Fragment &fragment) {
  if (fragment.fragmentType == FRAGMENTTYPE_OPTION)
    os << "Option ";
  else if (fragment.fragmentType == FRAGMENTTYPE_PARAM)
    os << "Param ";
  else {
    os << "Unknown fragmentType " << fragment.fragmentType << std::endl;
  }
  os << "optionName=" << (fragment.option? fragment.option->optionName: "(null)")
     << " argName=" << (fragment.option? fragment.option->argName: "(null)")
     << " optionValue=";
  if (!fragment.optionValue) {
    os << "(null)";
  }
  else
    os << fragment.optionValue;
  os << " argValue=" << (fragment.argValue? fragment.argValue: "(null)");
  return os;
}

void CmdLine::printError(std::ostream &os) {
  if (cmdspec->commandName.size())
    os << cmdspec->commandName << ": ";
  os << "Unrecognized option " << argv[paramIdx] << '\n';
}

